# File Maker for Unevenly Yellowed Photo Correction 32/16/8bit Version
# Ver. 5.53 Aug. 2024 32bit image support version (Bug Fixed)
# (c)2020-24 Ohnishi, Yasuo
from tokenize import Double
from xdrlib import ConversionError
from ij import IJ, ImagePlus
from ij import plugin
from ij import WindowManager as WM
from ij.io import OpenDialog
from ij.gui import GenericDialog
from ij.plugin import ChannelSplitter
from ij.plugin import Duplicator
from ij.process import ImageConverter, ImageProcessor, ByteProcessor,\
    ShortProcessor, FloatProcessor
from ij.plugin.filter import ThresholdToSelection
from loci.plugins import BF
from java.awt.event import ItemListener, AdjustmentListener
from java.awt import Color
import os
import datetime
from org.python.core import codecs
codecs.setDefaultEncoding('utf-8')

#------------------------
def set_Lmax(imp):
    # set Maximum value of Lightness
    bitdepth = imp.getBitDepth()
    if bitdepth == 32:
        L_max = imp.getDisplayRangeMax()
        if int(L_max) > 65536:
            L_max = 16777215
        elif int(L_max) > 256:
            L_max = 65535
        elif int(L_max) > 1.01:
            L_max = 255
        else:
            L_max = 1.0
    elif bitdepth == 16:
        L_max = 65535
    else: # 8 or 24 bit
        L_max = 255
    return L_max
#-----------------
def imageBinarySmpl(impOrg, lowerV, upperV, roiColor, maskname):
    # fill roiColor between lowerV - upperV
    # make binary mask image with setThreshold
    # lowerV: lowerThreshold / upperV: upperThreshold (double)
    # roiColor: Color Value, ROI filled with (double)
    # maskname: name of mask image (string)
    imp = Duplicator().run(impOrg)
    ip = imp.getProcessor()
    mask = ip.createProcessor(ip.width, ip.height) # Background Value = 0.0
    mask.setValue(roiColor)
                    # with which color ROI filled (normally L_max)
    ip.setThreshold(lowerV, \
        upperV, ImageProcessor.NO_LUT_UPDATE)
    roi = ThresholdToSelection.run(imp)
    mask.setRoi(roi)
    mask.fill(roi)
    maskimp = ImagePlus(maskname, mask)
    imp.close()
    return maskimp
#-----------------
def clipUnder0(imp, Mode, L_max = 1.0):
    ip = imp.getProcessor()
    if Mode != "32-bit":
        return imp
    ip_mask = FloatProcessor(ip.width, ip.height)
    ip_mask.setValue(0.0)
    ip_mask.fill()
    imp1 = ImagePlus("imp1", ip_mask)
    ic = plugin.ImageCalculator()
    imp2 = ic.run("Max created", imp, imp1) # bind masks
    imp2.setLut(None)
    imp2.show()
    imp2.hide()
    imp1.close()
    imp.hide()
    imp.close()
    return imp2
#--------------------
def clipOverLmax(imp, Lmax):
    # use only 32-bit mode
    ip = imp.getProcessor()
    ip_mask = FloatProcessor(ip.width, ip.height)
    ip_mask.setValue(Lmax)
    ip_mask.fill()
    imp1 = ImagePlus("imp1", ip_mask)
    ic = plugin.ImageCalculator()
    imp2 = ic.run("Min created", imp, imp1) # bind masks
    imp2.setDisplayRange(0.0, Lmax)
    imp1.close()
    imp.close()
    return imp2
#----------------------------------------------
def invert(imp, Mode, L_max):
    imp32 = Duplicator().run(imp)
    ImageConverter(imp32).convertToGray32()
    imp32.getProcessor().multiply(-1.0)
    imp32.getProcessor().add(L_max)
    if Mode == "8-bit":
        ImageConverter(imp32).convertToGray8()
    elif Mode == "16-bit":
        ImageConverter(imp32).convertToGray16()
    return imp32
#----------------------------------------------
def printStat(Stat, string):
    print (string + " mean", Stat.mean)
    print (string + " min", Stat.min)
    print (string + " max", Stat.max)
    print (string + " mode", Stat.mode)
#----------------------- 2022/7/5
def make_Gaussian_imp(imp_org, sigma, Mode):
    imp_width = imp_org.getWidth()
    imp_height  = imp_org.getHeight()
    swidth = int(imp_width / 4)
    sheight = int(imp_height / 4)
    imp_g = Duplicator().run(imp_org)
    if Mode == "8-bit":
        ImageConverter(imp_g).convertToGray32()
        # imp_g will be gauss image of level adjusted imp_org
    imp_g.resize(swidth, sheight, 'bilinear')
    IJ.run(imp_g, "Gaussian Blur...", "sigma=" + str(sigma))
    imp_g.resize(imp_width, imp_height, 'bilinear')
    if Mode == "8-bit":
        ImageConverter(imp_g).convertToGray8()
    imp_g.setDisplayRange(0.0, set_Lmax(imp_g))
    return imp_g
#------------------------------------
def get_cropped_imp_stat(imp):
    width = imp.getWidth()
    height = imp.getHeight()
    roi_x = int(width * 0.15)
    roi_y = int(height * 0.15)
    roi_width = int(width * 0.7)
    roi_height = int(height * 0.7)

    imp.setRoi(roi_x, roi_y, roi_width, roi_height)
    impTmp = imp.crop()
    stat = impTmp.getStatistics()
    impTmp.close()
    imp.resetRoi()
    return stat
#--------
# sub routine for pseud_Flat_field_C
def levelAdjust4GB(parm_leveladj, imp_g1, imp_b1, \
                    process, parm_correct_fac, Mode, L_max):
                                                # process == test
    """ imp_g1: G ch. Gaussan blur image
        imp_b1: B ch. Gaussan blur image """
    #if meanadjustment == True: # adjust level difference of G and B Ver. 3.53
        # modifyed in Ver. 5.3 / get statistics with cropped image
    stat_g1 = get_cropped_imp_stat(imp_g1)
    stat_b1 = get_cropped_imp_stat(imp_b1)
    if parm_leveladj == "Auto":
        if stat_g1.mean < stat_b1.mean:
            # adjust level difference of G and B Gaussian image (Ver. 3.52)
            ### Bug Fixed (Ver. 4.54) 2022_08_20
            imp_g1.getProcessor().add(stat_b1.mean - stat_g1.mean)
                                             # imp_g1 must be lighter
            if Mode == "32-bit":
                imp_g1 = clipOverLmax(imp_g1, L_max)
        elif (process == "B") and (stat_g1.mean > stat_b1.mean):
                                        # modification Ver. 4.55
            imp_g1.getProcessor().subtract(stat_g1.mean - stat_b1.mean)
            if Mode == "32-bit":
                imp_g1 = clipUnder0(imp_g1, Mode, L_max)
            # when subtract higher region of B than G and G is higher than B
            # also adjust level difference of G and B Gaussian image
    elif parm_leveladj == "Manual":
                # Manual adjustment of correction amount of B
        if stat_g1.mean < stat_b1.mean:
            # adjust level difference of G and B Gaussian image (Ver. 3.52)
            imp_g1.getProcessor().add(stat_b1.mean - stat_g1.mean)
        imp_g1.getProcessor().add(parm_correct_fac / 255.0 *L_max)
            # add correction value "parm_correct_fac"
            # modification in Ver. 4.55
            # make Manual value relative to auto adjusted value

#--------
# sub routine for pseud_Flat_field_C
def subtractHigherBthanG(imp_b1, imp_g1, imp_c_b0, Mode, L_max):
    # subtract higher region of B than G
    ic = plugin.ImageCalculator()
    imp_dif_bg = ic.run("Subtract create", imp_b1, imp_g1) # No window made
    stat = imp_dif_bg.getStatistics()
    if Mode == "32-bit":
        if stat.min < 0.0:
            imp_dif_bg = clipUnder0(imp_dif_bg, Mode, L_max)
    imp_dif_bg.setLut(None)
    imp_dif_bg.setTitle("imp_dif_bg")
    imp_dif_bg.show()
    #if debugmode == True:
    #    IJ.showMessage("imp_dif_bg")
    imp_dif_bg.hide()

    if Mode == "32-bit":
        stat = imp_dif_bg.getStatistics()
        if stat.min > 0.0:
            imp_dif_bg.setDisplayRange(stat.min, L_max)
        else:
            imp_dif_bg.setDisplayRange(0.0, L_max)

    imp_c_b = ic.run("Subtract create", imp_c_b0, imp_dif_bg) # No window made
    if Mode == "32-bit":
        stat = imp_c_b.getStatistics()
        if stat.min > 0.0:
            imp_c_b.setDisplayRange(stat.min, L_max)
        else:
            imp_c_b.setDisplayRange(0.0, L_max)
    imp_dif_bg.close()
    return imp_c_b


#--------
def pseudo_Flat_field_C(imp_g0, imp_b0, sigma, Mode, L_max, folder, \
        filename, parm_subtractBv, parm_leveladj, parm_correct_fac):
    debugmode = False  # select debugmode True or False /for debug
    global CountN
    CountN = CountN + 1
    """
        imp_g0: reference (G) ch. imp, imp_b0: target (B) ch. imp
        sigma: Gaussian blur sigma
        parm_subtractBv: Subtraction or not B when B is much higher than G
        parm_leveladj: adjust or not luminance level
        parm_correct_fac: manual correction factor
    """
    if Mode == "32-bit":
        imp_g0.setDisplayRange(0.0, L_max)
        stat = imp_b0.getStatistics()
        imp_b0.setDisplayRange(stat.min, L_max)
    ### Options for developping
    if parm_subtractBv == False:
        # No Subtraction B when B is much higher than G
        test = "A"  # Normal
    else:  # Subtraction B when B is much higher than G
        test = "B"  # optional
    if parm_leveladj != "No":
        meanadjustment = True # select mean adjustment True or False
    else:
        meanadjustment = False # select mean adjustment True or False
    ####
    imp_g1 = make_Gaussian_imp(imp_g0, sigma, Mode)
    imp_b1 = make_Gaussian_imp(imp_b0, sigma, Mode)
    if Mode == "32-bit":
        imp_g1.setDisplayRange(0.0, L_max)
        stat = imp_b1.getStatistics()
        imp_b1.setDisplayRange(stat.min, L_max)
    """ imp_g1: G ch. Gaussan blur image imp_b1: B ch. Gaussan blur image """
    if meanadjustment == True: # adjust level difference of G and B Ver. 3.53
        levelAdjust4GB(parm_leveladj, imp_g1, imp_b1, \
                    test, parm_correct_fac, Mode, L_max)
    if Mode == "32-bit":
        stat_g1 = get_cropped_imp_stat(imp_g1)
        stat_b1 = get_cropped_imp_stat(imp_b1)
        imp_g1.setDisplayRange(stat_g1.min, L_max)
        imp_b1.setDisplayRange(stat_b1.min, L_max)
    ic = plugin.ImageCalculator()
    imp_dif_gb = ic.run("Subtract create", imp_g1, imp_b1)
    stat = imp_dif_gb.getStatistics()
    if Mode == "32-bit":
        if stat.min < 0.0:
            imp_dif_gb = clipUnder0(imp_dif_gb, Mode, L_max)
        imp_dif_gb.setDisplayRange(0.0, L_max)
    imp_dif_gb.setLut(None)
    imp_dif_gb.setTitle("imp_dif_gb")
    imp_dif_gb.show()
    imp_dif_gb.hide()
    if Mode == "32-bit":
        stat = imp_dif_gb.getStatistics()
        if stat.min > 0.0:
            imp_dif_gb.setDisplayRange(stat.min, stat.max)
        else:
            imp_dif_gb.setDisplayRange(0.0, stat.max)
    imp_c_b0 = ic.run("Add create", imp_b0, imp_dif_gb)
    stat_imp_c_b0 = imp_c_b0.getStatistics()
    if debugmode == True:
        printStat(stat_imp_c_b0, "@psedFF Field imp_c_b0(0):")
        IJ.showMessage("@psedFF Field L_max: " + str(L_max))
    if Mode == "32-bit":
        if stat_imp_c_b0.min < 0.0:
            imp_c_b0 = clipUnder0(imp_c_b0, Mode, L_max)
    imp_c_b0.setLut(None)
    imp_c_b0.setTitle("imp_c_b0")
    imp_c_b0.show()
    imp_c_b0.hide()
                # image correction of B (subtract yellowed value from B)
    if Mode == "32-bit":
        stat_imp_c_b0 = imp_c_b0.getStatistics()
        if stat_imp_c_b0.min > 0.0:
            imp_c_b0.setDisplayRange(stat_imp_c_b0.min, L_max)
        else:
            imp_c_b0.setDisplayRange(0.0, L_max)
        if debugmode == True:
            printStat(stat_imp_c_b0, "@psedFF Field imp_c_b0:")
            IJ.showMessage("@psedFF Field imp_c_b0(2) L_max:" + str(L_max))
            stat_imp_c_b0 = imp_c_b0.getStatistics()
            printStat(stat_imp_c_b0, "@psedFF Field imp_c_b0(2):")

    if test =="B": #Optional Process
        # subtract higher region of B than G
        imp_c_b = subtractHigherBthanG(imp_b1, imp_g1, imp_c_b0, Mode, L_max)
    else:
        # No Subtraction B when B is much higher than G
        imp_c_b = Duplicator().run(imp_c_b0) # for test "A"
    stat_c_b = imp_c_b.getStatistics()
    if Mode == "32-bit":
        if stat_c_b.min > 0.0:
            imp_c_b.setDisplayRange(stat_c_b.min, L_max)
        else:
            imp_c_b.setDisplayRange(0.0, L_max)
    if Mode == "8-bit":
        ImageConverter(imp_c_b).convertToGray8()
    elif Mode == "16-bit":
        ImageConverter(imp_c_b).convertToGray16()
    elif Mode == "32-bit":
        ImageConverter(imp_c_b).convertToGray32()
    imp_c_b.setDisplayRange(0.0, L_max)
    imp_dif_gb.close()
    imp_c_b0.close()
    stat =imp_c_b.getStatistics()
    if Mode == "32-bit" and stat.max > L_max:
        imp_c_b.setDisplayRange(0.0, stat.max)
        imp_c_b = clipOverLmax(imp_c_b, L_max)
    stat_c_b = imp_c_b.getStatistics()
    return imp_c_b, imp_b1, imp_g1
#--------------------------
def make_imp_blueHigh(impRf2, impT, Mode, L_max):
    """
       making relatively high value region image of target channel
                                                    with user make mask
    impPrvWin: imp of preview window / impPrv: imp of preview image (ref. ch.)
    imp_binaryMask / impPrvBinMask: imp of user made mask image
    """
    # impT: target ch. image
    # impRf2: reference ch. image
    # read user made binary mask
    od = OpenDialog("Choose user made mask image file")
    folder = od.getDirectory()
    filename = od.getFileName()
    imps_Umask = BF.openImagePlus(folder + filename)
        # Open user made mask image file with Bio-format Plugin
    imp_binaryMask = imps_Umask[0] # user's binary mask Image
    stackSize = imp_binaryMask.getStackSize()
    if stackSize == 3: # if file image is RGB color...
        impTmp = imageConvert2Gray(imp_binaryMask)
        imp_binaryMask.close()
        imp_binaryMask = impTmp
    Mask_L_max = set_Lmax(imp_binaryMask)
    imp_binaryMask.setDisplayRange(0.0, Mask_L_max)

    # bit depth adjustment of original image and user's mask image
    ImageConverter(imp_binaryMask).setDoScaling(True)
    if Mode == "8-bit":
        ImageConverter(imp_binaryMask).convertToGray8()
    elif Mode == "16-bit":
        ImageConverter(imp_binaryMask).convertToGray16()
    elif Mode == "32-bit":
        ImageConverter(imp_binaryMask).convertToGray32()
    if L_max != Mask_L_max:
        imp_binaryMask.getProcessor().multiply(L_max / Mask_L_max)
            # adjustment for discrepancy of target image and user mask image
    # combine mask
    ic = plugin.ImageCalculator()
    impT_high0 =  ic.run("Min create", impT, imp_binaryMask)
    # impT_high0: masked target (B) ch. image
    impRf2_high0 =  ic.run("Min create", impRf2, imp_binaryMask)
    # impRf2_high0: masked reference (G) ch. image
    impT_high0.setDisplayRange(0.0, L_max)
    impRf2_high0.setDisplayRange(0.0, L_max)

    stat_impT_high0 = impT_high0.getStatistics()
        # statistics of target ch. (B)
    stat_impRf2_high0 = impRf2_high0.getStatistics()
        # statistics of ref. ch. (G)
    # making preview image for correct blue high region
    impPrv = make_preview(impRf2_high0, L_max)
    impPrvBinMask = make_preview(imp_binaryMask, L_max)
    impPrvWin = Duplicator().run(impPrv)
        # impPrvWin: image for preview image window
    impPrvWin.setTitle("High blue region correction preview")
    impPrvWin.show()
    ### input correction factor ###
    factor = gd_input_blue_high_cFfactor\
        (impPrvWin, impPrv, impPrvBinMask, stat_impT_high0)
            # Show dialog for set parameter
    # set parameter for ref. ch. highlight region image
    impPrvWin.hide()
    impPrvWin.close()
    impPrv.close()
    ### end of  input factor --
    #  adjust mask image (ref. ch.) with factor
    IJ.run(impRf2_high0, "Add...", \
            "value=" + str(stat_impT_high0.stdDev * factor))
    impRf2_high2 =  ic.run("Min create", impRf2_high0, imp_binaryMask)
    impRf2_high0.close()
    impRf2_high0 = impRf2_high2
    impRf2_high0.setTitle("ref image of blue high region")
    impRf2_high0.getProcessor().setLut(None)
    #---
    if stat_impRf2_high0.mean < stat_impT_high0.mean:
                # adjust Rf ch. value
        IJ.run(impRf2_high0, "Add...", \
            "value=" + str(stat_impT_high0.mean - stat_impRf2_high0.mean))
        impRf2_high1 =  ic.run("Min create", impRf2_high0, imp_binaryMask)
        impRf2_high0.close()
        impRf_highlight = impRf2_high1
    else:
        impRf_highlight = impRf2_high0
    impT_highlight  = impT_high0
    return impT_highlight, impRf_highlight
    # impT_highlight: target ch. highlight region image for pFFc
    # impRf_highlight: ref. ch. highlight region image for pFFc

#--------------------------------
def gd_input_blue_high_cFfactor\
        (impPrvWin, impPrv, impPrvBinMask, statT):
    """
    impPrvWin: imp of preview window / impPrv: imp of preview image (ref. ch.)
    impPrvBinMask: imp of user made mask image
    statT: statistics of target ch.
    """
    gd = \
        GenericDialog("Input correction factor for bluer region")
    gd.addSlider("Correction factor (%) x std Dev", 0, 300, 0)
    gd.addMessage("Normally 100 (%) is recommended.")
    #-- define UI elements ---
    slider_factor = gd.getSliders().get(0)
    nfield_slider_factor = gd.getNumericFields().get(0)
    #-- set previewer --------
    previewer = imagePreviewer(impPrvWin, impPrv, impPrvBinMask, \
                    slider_factor, nfield_slider_factor, statT)
    slider_factor.addAdjustmentListener(previewer)
    #-------------------------
    gd.showDialog()
    bluer_factor = gd.getNextNumber()/100
    return bluer_factor
#--------------------------------
class imagePreviewer(AdjustmentListener):
    def __init__(self, impPrvWin, impPrv, impPrvBinMask, \
                slider_factor, nfield_slider_factor, statT):
        """
            imp*: an ImagePlus
            slider_*: a java.awt.Scrollbar UI element
        """
        self.impPrvWin = impPrvWin # imp of preview windows
        self.impPrv = impPrv # preview image (ref. ch.)
        self.impPrvBinMask = impPrvBinMask # mask image
        self.slider_factor = slider_factor
        self.nfield_slider_factor = nfield_slider_factor
        self.stat = statT # statistics of target ch. (B)

    def adjustmentValueChanged(self, event):
        """
        event: an AdjustmentEvent with data on the state of the scroll bar.
        """
        if (IJ.isMacOSX() == False) and event.getValueIsAdjusting():
            return
                # there are more scrollbar adjustment events queued already
        self.updatePrView()

    def updatePrView(self):
        factor = float(self.nfield_slider_factor.getText()) / 100.0
        impPrv = Duplicator().run(self.impPrv)
        impPrv.getProcessor().add(self.stat.stdDev * factor)
                            #stdDev must be calculated before masking?
        ic = plugin.ImageCalculator()
        impPrv1 =  ic.run("Min create", impPrv, self.impPrvBinMask)
        ip = impPrv1.getProcessor()
        self.impPrvWin.setProcessor(ip)
        # self.impPrvWin: canvas for preview image
        impPrv.close()
        impPrv1.close()
        self.impPrvWin.updateAndDraw()
#--------------------------------
def make_preview(imp, L_max):
    """ making preview image """
    impprv = Duplicator().run(imp) # impprv: preview imp
    impprv.setDisplayRange(0.0, L_max)
    scale = 1280.0 / impprv.getWidth()
                    # preview image scaling value (width = 1200px fixed)
    new_height = impprv.getHeight() * scale # preview image height
    new_height = int(new_height)
    impprv1 = impprv.resize(1280, new_height, "none")
    impprv1.getProcessor().setLut(None)
    return impprv1
#----------------------------------
#---mask for excluding more bluer region from correction (for Hybrid correction)
def make_blue_mask(impG, impB, adjstV, inverse, Mode, L_max):
    # This routine is making mask for foreground layer
    debugmode = False
    ic = plugin.ImageCalculator()
    imp8_B = Duplicator().run(impB) # corrected B ch.
    imp8_G = Duplicator().run(impG) # G ch.
    # modifyed in Ver. 5.3 / get statistics with cropped image
    statB = get_cropped_imp_stat(impB)
    statG = get_cropped_imp_stat(impG)
    if statG.mean < statB.mean: #adjust B value relative to G
        IJ.run(imp8_G, "Add...", "value=" + str(statB.mean - statG.mean))
                                            # del int 4.82b
    imp8_G.getProcessor().subtract(adjstV / 255.0 * L_max) # adjusting blue threshold
    imp8_B_G = ic.run("Subtract create", impB, imp8_G)
    stat = imp8_B_G.getStatistics()
    if Mode == "32-bit":
        if stat.min < 0.0:
            imp8_B_G = clipUnder0(imp8_B_G, Mode, L_max)
        imp8_B_G.setDisplayRange(0.0, L_max)
        if stat.min < 0.0:
            imp8_B_G = clipUnder0(imp8_B_G, Mode, L_max)
    imp8_G.close()
        # extract more green region
    stat = imp8_B_G.getStatistics()
    imp8_B_G.getProcessor().multiply(7.0) # adjusting transparency
    imp8_B_G.setDisplayRange(0.0, L_max)
    if Mode == "32-bit":
        imp8_B_G = clipOverLmax(imp8_B_G, L_max)
        imp8_B_G.setDisplayRange(0.0, L_max)
    if inverse == True:
        IJ.run(imp8_B_G, "Invert", "") # bluer region  > blacker (opaque)
                                # not to apply correction in bluer region
    imp8_0 = imp8_B_G
    stat = imp8_0.getStatistics()
    imp8_0.setDisplayRange(0.0, L_max)
    if imp8_0.getBitDepth() == 32:
        if stat.min < 0.0:
            imp8_0 = clipUnder0(imp8_0, "32-bit", L_max)
        elif stat.max > L_max:
            imp8_0 = clipOverLmax(imp8_0, L_max)
    IJ.run(imp8_0, "Gamma...", "value=0.5")
    IJ.run(imp8_0, "Gaussian Blur...", "sigma=1")
    imp8_0.setDisplayRange(0.0, L_max)
    imp8_B.close()
    imp8_G.close()
    return imp8_0
    #imp8_0: region bluer > more opaque (not to apply correction)
#--------------------
def making_greenC_layer(imp, L_max, Mode):
    # imp: fg correcting layer
    debugmode = False
    if debugmode == True:
        IJ.showMessage(str(L_max) + " / " + Mode)
        stat = imp.getStatistics()
        mean = stat.mean  / (L_max / 255)
        max = stat.max  / (L_max / 255)
        IJ.showMessage("imp: mean / max: " + str(mean) + " / " + str(max))
    impShadow = Duplicator().run(imp)
    impMidHigh = Duplicator().run(imp)
    #-- shadow part ---
    shadowFactor = 0.5
    impShadow.getProcessor().multiply(shadowFactor)
    ipShadow = impShadow.getProcessor()
    impShadow.getProcessor().setThreshold(0.125 * shadowFactor * L_max, \
        L_max, ImageProcessor.NO_LUT_UPDATE)
    roi = ThresholdToSelection.run(impShadow)
    # fill white over 0.125 * L_max
    ipShadow.setValue(L_max) # White
    ipShadow.setRoi(roi)
    ipShadow.fill(roi)
    impShadow.setProcessor(ipShadow)
    #-- mid-highlight part ---
    impMidHigh.getProcessor().subtract(0.125 * L_max)
    if Mode == "32-bit":
        impMidHigh = clipUnder0(impMidHigh, Mode, L_max)
    midHighFactor = 0.5
    impMidHigh.getProcessor().multiply(midHighFactor)
    #impMidHigh.setDisplayRange(0.0, (1 - 0.125) * midHighFactor * L_max)
    impMidHigh.setDisplayRange(0.0, L_max)
    if debugmode == True:
        stat = impMidHigh.getStatistics()
        mean = stat.mean  / (L_max / 255)
        max = stat.max  / (L_max / 255)
        IJ.showMessage("impMidHigh 1: mean / max: " + str(mean) + " / " + \
        str(max)+ " / " + str(impMidHigh.getDisplayRangeMax()/ (L_max / 255)))
    impMidHigh.getProcessor().gamma(1.0 / 1.5)
    if debugmode == True:
        stat = impMidHigh.getStatistics()
        mean = stat.mean  / (L_max / 255)
        max = stat.max  / (L_max / 255)
        IJ.showMessage("impMidHigh 2: mean / max: " + str(mean) + " / " + str(max))
    impMidHigh.setDisplayRange(0.0, L_max)
    impMidHigh.getProcessor().add(0.125 * shadowFactor * L_max)
    #-- bind two images ---
    ic = plugin.ImageCalculator()
    imp1 = ic.run("Min created", impShadow, impMidHigh) # bind masks
    if debugmode == True:
        stat = imp1.getStatistics()
        mean = stat.mean  / (L_max / 255)
        max = stat.max  / (L_max / 255)
        IJ.showMessage("imp: mean / max: " + str(mean) + " / " + str(max))
    """imp1.show()
    IJ.showMessage("")
    imp1.hide()"""
    return imp1
#---mask for preserving plants green area from correction
#                                                (for pseudo FF correction)
def make_green_prsv_mask_new(impG, imp_c_b, impB, impR, Mode, L_max, \
                                            parm_Binary, folder, filename):
    """
    This routine is for preserving plants green option of foreground
                layer mask in Hybrid / pseudo flat-field correction algorism
    impG: G ch. / imp_c_b: corrected B ch. / impB: original B ch. / impR: R ch.
    imp_R128: mask for background correction mask (R128) > but no use
    parm_Binary: imp_R128's threshold > but no use
    #!!!!!! imp_c_b may context adjusted B
    """
    debugmode = False
    ic = plugin.ImageCalculator()
    imp8_cB = Duplicator().run(imp_c_b) # corrected B ch.
    imp8_G = Duplicator().run(impG) # G ch.
    imp8_R = Duplicator().run(impR) # R ch.
    imp8_Brestriction = Duplicator().run(impB) # B ch.
    imp8_cB.setDisplayRange(0.0, L_max)
    imp8_G.setDisplayRange(0.0, L_max)
    imp8_R.setDisplayRange(0.0, L_max)
    imp8_Brestriction.setDisplayRange(0.0, L_max)
    # making exclucing region image
    Threshold = parm_Binary # changed in 4.82
        # threshold of highlight / midtone
    #cThreshold = Threshold / 255.0 * L_max
    Threshold = Threshold + 25.5
    imp8_B130 = make_bg_mask(imp8_cB, Threshold, Mode, L_max)
    imp8_B130.getProcessor().invert()
    imp8_B130.setDisplayRange(0.0, L_max)
        # ROI (under 130) will be filled with white
        # under 130 > white and over 130 > black

    # making relative green mask process start
    stat8_cB = imp8_cB.getStatistics()
    statR = impR.getStatistics()
    # level adjustment of R and G
    stat8_G = imp8_G.getStatistics()
    if stat8_G.mean < statR.mean: #adjust R value relative to G
        IJ.run(imp8_G, "Add...", "value=" + str(statR.mean - stat8_G.mean))
                                                    # del int 4.82b
    # making reference image to make relative green area image
    # making R x 0.2 + corrected B x 0.8 image
    imp8_R.getProcessor().multiply(0.30) ## variable?
                            # image R x 0.20 #*** 2023/1/5 revised
    imp8_cB.getProcessor().multiply(0.70) # image B x 0.8
    imp8_RandB = ic.run("Add create", imp8_R, imp8_cB)
    stat = imp8_RandB.getStatistics()
    if Mode == "32-bit":
        if stat.min < 0.0:
            imp8_RandB = clipUnder0(imp8_RandB,  "32-bit", L_max)
        if stat.max > L_max:
            imp8_RandB = clipOverLmax(imp8_RandB, L_max)
    imp8_RandB.setDisplayRange(0.0, L_max)
        # imp8_RandB: R and B mixing image
    # G - (R x 0.25 + B x 0.75) for making yellowish green sensitive mask
    IJ.run(imp8_G, "Add...", "value=" + str((15 / 255.0) * L_max))
    stat = imp8_G.getStatistics()
    """if Mode == "32-bit" and stat.max > L_max:
        imp8_G = clipOverLmax(imp8_G, L_max)"""
    # making relative green area image / G ch. - ref. image
    imp8_G_cB = ic.run("Subtract create", imp8_G, imp8_RandB)
    stat = imp8_G_cB.getStatistics()
    if Mode == "32-bit":
        if stat.min < 0.0:
            imp8_G_cB = clipUnder0(imp8_G_cB, "32-bit", L_max)
        # imp8_G_cB: relative G iamge (relative Green area = lighter)
    factor = L_max / (stat.mean + stat.stdDev * 2)
    imp8_G_cB.getProcessor().multiply(factor)
    imp8_G_cB.setDisplayRange(0.0, L_max)
    imp8_G_cB.getProcessor().gamma(1.0 / 2.6)  ## variable?

        # data clip for 32bit image
    if debugmode == True:
        IJ.saveAsTiff(imp8_B130, folder + filename + "_imp8_B130.tif")
        IJ.saveAsTiff(imp8_G_cB, folder + filename + "_imp8_G_cB.tif")
    # clopping over mid gray point area from relative green image
    imp8b = ic.run("Min create", imp8_G_cB, imp8_B130)
    if Mode == "32-bit":
        if stat.min < 0.0:
            imp8b = clipUnder0(imp8b, "32-bit", L_max)
    imp8b.setDisplayRange(0.0, L_max)
        # excluding over 130 (Threshold) area
        # under 130: relative G image / over 130: black
    # deciding multiplying factor
    stat8b = imp8b.getStatistics()
    mult_factor = L_max / stat8b.max
    if debugmode == True:
        IJ.showMessage("stat8b.max: " + str(stat8b.max))
        IJ.showMessage("stat8b.min: " + str(stat8b.min))
        IJ.showMessage("mult_factor: " + str(mult_factor))
#    mult_factor = mult_factor * 2.0
        #culculate multiplying factor
    IJ.run(imp8b, "Multiply...", "value=" + str(mult_factor))
                                        # mask x multiplying factor
    stat8b = imp8b.getStatistics()
    if Mode == "32-bit":
        if stat8b.max > L_max:
            imp8b = clipOverLmax(imp8b, L_max)
    imp8b.setDisplayRange(0.0, L_max)
    stat8b = imp8b.getStatistics()
    if debugmode == True:
        IJ.showMessage("inverted stat8b.max: " + str(stat8b.max))
        IJ.showMessage("inverted stat8b.min: " + str(stat8b.min))
        # more green region  > more black (opaque) and over region 130 > white

    #----------------------------------------------------------------
    imp8b.getProcessor().gamma(1.0 / 4.0)
    imp8b.setDisplayRange(0.0, L_max)
    imp8_cB.close()
    imp8_R.close()
    imp8_G_cB.close()
    imp8_B130.close()
    imp8b.close()
    imp8_Brestriction.close()
    if Mode == "8-bit":
        ImageConverter(imp8b).convertToGray8()
    elif Mode == "16-bit":
        ImageConverter(imp8b).convertToGray16()
    return imp8b
    """
    imp8b:
        region relatively green > more opaque
    """
#-----Image Convert to Gray-------------
# 3 channel color image to mono channel grayscale image / Ver. 4.71
def imageConvert2Gray(imp):
    Lmax = set_Lmax(imp)
    impsCh2 = ChannelSplitter.split(imp) #Channel Splitting of imp0
    ic = plugin.ImageCalculator()
    for i in impsCh2:
        if i == 0:
            IJ.run(i, "Multiply...", "value=0.299") # R
        elif i ==1:
            IJ.run(i, "Multiply...", "value=0.587") # G
        elif i ==2:
            IJ.run(i, "Multiply...", "value=0.114") # B
    impTmp0 = ic.run("Add create", impsCh2[0], impsCh2[1])
    impTmp0.setDisplayRange(0.0, Lmax)
    impTmp1 = ic.run("Add create", impTmp0, impsCh2[2])
    impTmp1.setDisplayRange(0.0, Lmax)
    impTmp0.close()
    return impTmp1

#---------------------------------
def preMixing(impT, impPrMix, parm_FFC_sigma, parm_correct_fac, Mode, \
                                                    L_max, folder, filename):
    # add parm_correct_fac Ver. 4.71
    """
        This routine is for premixing G channel to B for original B image
        impT: Target Ch.  impPrMix: Pre mixing Ch.
    """
    debugMode = False
    # imp_c_Mx: modified mixing G ch.
    # in arg. pseudo_Flat_field_C below: target: impPrMix(G) ref: impT(B) 
    # it's opposite to normal!! Aug. 2024
    imp_c_Mx, imp_Gauss_preMix1, imp_Gauss_T1 = \
        pseudo_Flat_field_C(impT, impPrMix, parm_FFC_sigma * 4, Mode, L_max, \
                            folder, filename, True, "Auto", parm_correct_fac)
    # if parm_FFC_sigma is larger, then image quality will
    #                                   be much better for pre mixing
    # make adjusted G ch.
    # imp_c_Mx: corrected Mixing ch.
    #   parm_subtractBv: True, parm_leveladj: "Auto"
    impT_tmp03 = Duplicator().run(impT) # B ch.
    # modifyed in Ver. 5.3 / get statistics with cropped image > deleted (V. 5.52)
    IJ.run(impT_tmp03, "Multiply...","value=0.3")
    IJ.run(imp_c_Mx, "Multiply...","value=0.7")
    if Mode == "32-bit":
        impT_tmp03.setDisplayRange(0.0, L_max * 0.3)
        imp_c_Mx.setDisplayRange(0.0, L_max * 0.7)
    impT.close()
    ic = plugin.ImageCalculator()
    impT = ic.run("Add create", impT_tmp03, imp_c_Mx)
    impT.setDisplayRange(0.0, L_max)
    # output for bugfixing
    if debugMode == True:
        IJ.saveAsTiff(imp_c_Mx, folder + filename + "_prMixingR.tif")
    #--
    impT_tmp03.close()
    imp_c_Mx.close()
    imp_Gauss_preMix1.close()
    imp_Gauss_T1.close()

    return impT

#-----------------
def extractDarkArea(darkThreshold, imp, Mode, L_max): #there are bugs
    # attention!: imp's max value may be higher than L_max
    imp1 = Duplicator().run(imp)
    imp1.setDisplayRange(0.0, L_max)
    ip1 = imp.getProcessor()

    if Mode == "32-bit":
        blank_mask = FloatProcessor(ip1.width, ip1.height)
    elif Mode == "16-bit":
        blank_mask = ShortProcessor(ip1.width, ip1.height)
    else:
        blank_mask = ByteProcessor(ip1.width, ip1.height)
    blank_mask.BLACK_AND_WHITE_LUT
    blank_mask.setValue(L_max) # ROI: L_max / otherwise: 0.0
    blank_mask.fill()
    impDarkMask = ImagePlus("Dark parts layer mask", blank_mask)

    ip2 = impDarkMask.getProcessor()

    imp1.getProcessor().setThreshold(((darkThreshold / 255.0) * L_max), \
        L_max, ImageProcessor.NO_LUT_UPDATE)
    print ("Dark Threshold ", darkThreshold, "=", \
           (darkThreshold / 255.0) * L_max, L_max)
    roi = ThresholdToSelection.run(imp1)
    # Dark Layer image
    ip1.setValue(L_max)
    ip1.setRoi(roi)
    ip1.fill(roi)
    imp1.setProcessor(ip1)
    # Mask for Dark Layer image
    ip2.setValue(0)
    ip2.setRoi(roi)
    ip2.fill(roi)
    impDarkMask.setProcessor(ip2)

    return imp1, impDarkMask

#-----------------------------------------
def outputImg(imp, TRCType):
    if TRCType == "Linear":
        impOut = sRGB_TRC_decode(imp, L_max)
        #impOut = Duplicator().run(imp)
    else:
        impOut = Duplicator().run(imp)
    return impOut

#-----------------------------------------
def RGBmanipulation(folder, filename, parm, imp0, L_max, Mode, TRCType):
    #IJ.showMessage("Mode: " + str(Mode) + " L_max: " + str(L_max))
    debugMode = False
#  input Parameters
    if parm is not None:
 # if parm is not null object, then...
        parm0, parm_Binary, parm_Gain, parm_Shadow, Show, parm_Prph, \
            parm_Fg_tL, bool_PFFcOnly, parm_FFC_sigma, parm_Gmixrate, \
            parm_subtractBv, parm_leveladj, parm_correct_fac, HybridC, \
            Fg_mask_type, bgG_gamma, parm_ReadMask, DoPreMixing = parm
        print(Mode, parm_Binary, parm_Gain, parm_Shadow, Show, parm_Prph, \
                                                                parm_Fg_tL)
#     Mode: Image bit depth
#     L_max: Maximum Value of imp0
#     TRCType: Type of TRC
#     Show: Show working windows or not
#     parm_Binary: Threshold of fore / background
#     parm_Gain: Lightness of Distance adjusting Layer
#     parm_Shadow: Threshold of Dark part adjustment image
#     parm_Prph: Threshold of Periphery adjustment image
#     parm_Fg_tL: Threshold of Foreground mask
#     parm_Gmixrate: G mixing rate for foreground image in EPFFC only mode
#     parm_subtractBv: execute value subtraction of B value higher region
#     parm_leveladj: execute level adjustment G and B and it's type: Auto/Manual/None
#     parm_correct_fac: Manual correction factor
#     HybridC: execute hybrid correction or not and it's type
#     bgG_Gamma: Gamma value of mixing G for background image
#     Fg_mask_type: Type of mask for foreground layer Type1 to 4 and None
#     parm_ReadMask: read user made mask True/False
#     DoPreMixing: Do Premixing G to B for original B image True/False
# Always Open file with Bio-format Plugin
    #imp0 = imps[0] #imp0: Original Image
    #-- adjustment Sigma of Extended pseudo FF correction width image pixel size (4.60c)
    imp0_width = imp0.getWidth()
    imp0_height =  imp0.getHeight()
    imp0_long = max(imp0_width, imp0_height)
    sigma_factor = 10 / ((imp0_long // 1000) + 1)
    parm_FFC_sigma = parm_FFC_sigma / sigma_factor
    #--

    if Mode == "32-bit":
        IJ.showMessage("32-bit image Lightness Max Value: " + str(L_max))
    imp0.setDisplayRange(0.0, L_max)
    if Show == "Yes":
    # Windows Showing Mode
        imp0.show()
    #------
    impsCh0 = ChannelSplitter.split(imp0) #Channel Splitting of imp0
    for i in impsCh0: # it needs to convert Grayscale image
        if Mode == "8-bit":
            ImageConverter(i).convertToGray8()
        elif Mode == "16-bit":
            ImageConverter(i).convertToGray16()
        else:
            ImageConverter(i).convertToGray32()
        i.setDisplayRange(0.0, L_max)
    ######################
    if Mode != "8-bit":
        impsCh = B_level_adjustment(impsCh0, Mode, L_max, TRCType)
        for i in impsCh0:
            i.close()
    else:
        impsCh = impsCh0
    ######################
    ## R ch.
    impR =  impsCh[0]
    IJ.saveAsTiff(impR, folder + filename + "_R.tif")
    if TRCType == "Linear":
        impR = sRGB_TRC_encode(impR, L_max)
    ## G ch.
    impG =  impsCh[1]
    IJ.saveAsTiff(impG, folder + filename + "_G.tif")
    if TRCType == "Linear":
        impG = sRGB_TRC_encode(impG, L_max)
    ## B ch. & B x 0.3
    impB = Duplicator().run(impsCh[2])
    if DoPreMixing == False:
        IJ.saveAsTiff(impB, folder + filename + "_B.tif")
        if TRCType == "Linear":
            impB = sRGB_TRC_encode(impB, L_max)
    else: # execute premixing G to B
        if TRCType == "Linear":
            impB = sRGB_TRC_encode(impB, L_max)
        imp_nB = preMixing(impB, impG, parm_FFC_sigma / 3, \
                           parm_correct_fac, Mode, L_max, folder, filename)
        imp_nBout = outputImg(imp_nB, TRCType)
        impBout = outputImg(impB, TRCType)
        IJ.saveAsTiff(imp_nBout, folder + filename + "_B.tif")
        IJ.saveAsTiff(impBout, folder + filename + "_Borg.tif")
        imp_nBout.close()
        impBout.close()
        impB.close()
        impB = imp_nB
        # replace impB with premixing & corrected B image
    return impR, impG, impB
#---------------------
def base(impR, impG, impB, folder, filename, parm, L_max, Mode, TRCType):
    debugMode = False
    if parm is not None:
 # if parm is not null object, then...
        parm0, parm_Binary, parm_Gain, parm_Shadow, Show, parm_Prph, \
            parm_Fg_tL, bool_PFFcOnly, parm_FFC_sigma, parm_Gmixrate, \
            parm_subtractBv, parm_leveladj, parm_correct_fac, HybridC, \
            Fg_mask_type, bgG_gamma, parm_ReadMask, DoPreMixing = parm
        print(Mode, parm_Binary, parm_Gain, parm_Shadow, Show, parm_Prph, \
                                                                parm_Fg_tL)
    # Show Windows
    if Show == "Yes":
        impR.show()
        impG.show()
        impB.show()
    #--- preparetion for making layer images / to make (R+G)/2 image
    #         & periphery correction layer
    # G ch. x 0.5 / Make impG05  as G x 0.5 Image
    impG05 = Duplicator().run(impG)
    if Show == "Yes":
        impG05.show()
    IJ.run(impG05, "Multiply...", "value=0.5")
    if Mode == "32-bit":
        impG05.setDisplayRange(0.0, L_max * 0.5)
    # R ch. x 0.5 / Make impR05  as R x 0.5 Image
    impR05 = Duplicator().run(impR)
    if Show == "Yes":
        impR05.show()
    IJ.run(impR05, "Multiply...", "value=0.5")
    if Mode == "32-bit":
        impR05.setDisplayRange(0.0, L_max * 0.5)
    # B x 0.5 + R x 0.5 -- make peripherry corrction layer
    ic = plugin.ImageCalculator()
    imp5 = ic.run("Add create", impR05, impG05) # No window made
    imp5.setDisplayRange(0.0, L_max)

    #-----save periphery correction layer (no pseudo FF correction)
    if (bool_PFFcOnly == False) and (HybridC == "Off"):
        # traditional algorism
        imp5.getProcessor().setLut(None)
        imp5out = outputImg(imp5, TRCType)
        IJ.saveAsTiff(imp5out, folder + filename + "_Periph_Adjst.tif")
        imp5out.close()
    #-----save periphery correction layer (pseudo FF correction)
    else: # Hybrid & EPFFc
        imp_periph, imp_Gauss_b1, imp_Gauss_rg1 = \
            pseudo_Flat_field_C(imp5, impB, parm_FFC_sigma, Mode, L_max, \
                    folder, filename, True, parm_leveladj, parm_correct_fac)
        if debugMode == True:
            IJ.saveAsTiff(imp_Gauss_b1, folder + filename + \
                                    "_Periph_Adjst_Gauss_b1.tif")
            IJ.saveAsTiff(imp_Gauss_rg1, folder + filename + \
                                    "_Periph_Adjst_Gauss_rg1.tif")
        imp_Gauss_b1.close()
           # do pseudo FF correction with option subtract B higher region
           # reference image must be impG or imp5?
        imp_periph.setDisplayRange(0.0, L_max)
        imp_periph.getProcessor().setLut(None)
        imp_periphout = outputImg(imp_periph, TRCType)
        IJ.saveAsTiff(imp_periphout, folder + filename + "_Periph_Adjst.tif")
        imp_periphout.close()
        imp_periph.close()
    if Show == "Yes": #Hide the image windows to close
        imp5.show()
        impR05.hide()
        impG05.hide()
    impR05.close()
    impG05.close()

    #--- making foreground layer image
    imp3, impB03, imp_c_b = making_Fg_image(impG, impB, parm_FFC_sigma, \
        HybridC, Mode, L_max, folder, filename, parm_subtractBv, \
        parm_leveladj, parm_correct_fac, bool_PFFcOnly, parm_Gmixrate, \
        parm_ReadMask, Show, debugMode)
        # imp3: corrected foreground layer image
        # impB03: imp3 x 0.3 image
        # imp_c_b: extended pseudo flat-field correction applyed B
    imp3.setDisplayRange(0.0, L_max)
    imp3.getProcessor().setLut(None)
    imp3out = outputImg(imp3, TRCType)
    IJ.saveAsTiff(imp3out, folder + filename + "_B_Fg.tif")
    imp3out.close()
        # even in new alogorithm G mixing is needed to improve texture
    if debugMode == True:
        IJ.saveAsTiff(imp_c_b, folder + filename + "_imp_c_B.tif")
        # even in new alogorithm G mixing is needed to improve texture
        #--- making fg brown correction layer
    impBrownC = making_brownC_layer(imp3, Mode, L_max)
    impBrownCout = outputImg(impBrownC, TRCType)
    IJ.saveAsTiff(impBrownCout, folder + filename + "_Brown_Adjst.tif")
    impBrownCout.close()
    impBrownC.close()

    # --- making fg brown correction layer mask ----------
    impBrownMask = making_diff_RG_mask(imp3, impR, impG, parm_Binary, \
                                       L_max, Mode, folder, filename)
    IJ.saveAsTiff(impBrownMask, folder + filename +  \
                                    "_Brown_Adjst_Mask.tif")
    impBrownMask.close()
    #-------------
    #imp3.close()

    #--- making background layer image
    imp4 = making_Bg_image(impG, impB03, imp_c_b, bool_PFFcOnly, \
                        128, bgG_gamma, parm_Gain, Show, L_max) # 128: parm_Fg_tL
        #imp4: background layer image
    imp4.setDisplayRange(0.0, L_max)
    imp4.getProcessor().setLut(None)
    imp4out = outputImg(imp4, TRCType)
    IJ.saveAsTiff(imp4out, folder + filename + "_B_Bg.tif") #Bug Fix 2024_06
    imp4out.close()
    imp4.close()
    if Show == "Yes":
        impB03.hide()
    impB03.close()
    #--- making Periphery Adjustment Mask
    # Periphery Adjustment Mask: B - ((R+G)/2-10)
    imp2GR = making_Prph_adj_Mask(imp5, impB, parm_Prph, \
                                bool_PFFcOnly, Mode, L_max, Show)
        # imp2GR: Periphery adjustment mask image
    imp2GR.setDisplayRange(0.0, L_max)
    imp2GR.getProcessor().setLut(None)
    IJ.saveAsTiff(imp2GR, folder + filename + "_Periph_Adjst_Mask.tif")
    if Show == "Yes":
    #Hide the image windows to close
        imp2GR.hide()
    # close the image
    imp2GR.close()

    #--- making dark parts correction layer
    # Dark Parts 0 - parm_Shadow
    imp6, impDarkMask = extractDarkArea(parm_Shadow, imp5, Mode, L_max)
    imp_dark = imp6

    imp_dark.setDisplayRange(0.0, L_max)
    impDarkMask.setDisplayRange(0.0, L_max)

    imp_dark.getProcessor().setLut(None)
    impDarkMask.getProcessor().setLut(None)
    imp_darkout = outputImg(imp_dark, TRCType)
    IJ.saveAsTiff(imp_darkout, folder + filename + "_Shadow.tif")
    imp_darkout.close()
    IJ.saveAsTiff(impDarkMask, folder + filename + "_Shadow_Mask.tif")
    if Show == "Yes":
    #Hide the image windows to close
        imp5.hide()
        imp6.hide()
    imp5.close()
    #Close the images
    imp6.close()
    #--- making background correction mask
    imp9 = make_bg_mask(impR, parm_Binary, Mode, L_max) # Ver. 5.3a
                            # Make Bg Mask: black for under threshold
                            # parm_Binary: Bg Layer threshold
                            # Bg Threshold +/-20 > gradient (in Ver. 5.3a)
    imp9.setDisplayRange(0.0, L_max)
    if Show == "Yes":
        imp9.show()
    imp9.getProcessor().setLut(None)
    IJ.saveAsTiff(imp9, folder + filename + "_Bg_Thld_Mask.tif")

    #--- making foreground correction mask
    # R + parm_Fg_tL(128) ## mask image for foreground correction layer
    Fg_mask_type = "Type2" # temp setting in Nov. 2023
    adjstV = 0.0 # blue threshold
    imp8 = make_blue_mask(impG, impB, adjstV, True, Mode, L_max) #True<Inverse
        # imp8: Fg layer mask image
    if Show == "Yes":
        imp8.show()
    if debugMode == True:
        IJ.showMessage("Mask Type: " + Fg_mask_type)
    imp8.setDisplayRange(0.0, L_max)
    imp8.getProcessor().setLut(None)
    IJ.saveAsTiff(imp8, folder + filename + "_Fg_Mask.tif")
    #--- making Fg mask suppliment image for yellow parts ----
    #------ for reserving editing of yellow parts (from ver. 5.5)
    process = "B"
    impRtmp = Duplicator().run(impR)
    if process == "A": # A or B
        statR = impR.getStatistics()  
        stat3 = imp3.getStatistics()
        if statR.mean < stat3.mean:
            impRtmp.getProcessor().add(stat3.mean - statR.mean)
        impRinv =  ic.run("Subtract create", impRtmp, imp3) # relativ R image
        statImpR_3 = impRinv.getStatistics()
        impRtmp.close()
        impRinv.show()
        IJ.showMessage(str(statImpR_3.max))
        impRinv.hide()
        factor = (L_max / statImpR_3.max) * 2
        factor = 10.0
        impRinv.getProcessor().multiply(factor)
        IJ.saveAsTiff(impRinv, folder + filename + "_impR-imp3.tif")
    else:
        impRinv = impRtmp
    impRinv.getProcessor().invert()
    statRinv = impRinv.getStatistics()
    if statRinv.min > 0.0:
        impRinv.getProcessor().subtract(statRinv.min)
    IJ.saveAsTiff(impRinv, folder + filename + "_impR-imp3_inv.tif")
    imp8Yreserve = ic.run("Min create", imp8, impRinv)
    IJ.saveAsTiff(imp8Yreserve, folder + filename + "_Fg_Mask_Yrsv.tif")
    impRinv.close()
    imp8Yreserve.close()
    if Show == "Yes":
        imp8.hide()
    imp8.close()

    #--- making green correction layer
    #impGreen = making_greenC_layer(imp3, L_max, Mode)
    #impGreen.getProcessor().multiply(0.6) # darker adjustment
                        #imp3: Fg correction image (imp_c_b mixed with G)
    impGreen = Duplicator().run(impB)
    impGreenout = outputImg(impGreen, TRCType)
    IJ.saveAsTiff(impGreenout, folder + filename + "_Green_Adjst.tif")
    impGreenout.close()
    impGreen.close()
    #--- making green correction layer mask ---
    imp8b = make_green_prsv_mask_new(impG, imp_c_b, impB, impR, \
                Mode, L_max, parm_Binary, folder, filename) # change 4.71
    imp8b.setDisplayRange(0.0, L_max)
    adjstV = 35
    impBlmask = \
        make_blue_mask(impG, impB, adjstV, True, Mode, L_max) # inverse = True
    ic = plugin.ImageCalculator()
    impGreenC_mask =  ic.run("Min create", imp8b, impBlmask) # Nov. 2023
    IJ.saveAsTiff(impGreenC_mask, folder + filename + "_Green_Adjst_Mask.tif")

    imp_c_b.close() # pseud FFc image of B
    imp3.close() # Fg correction image
    #---------------------
    if Show == "Yes":
    #Hide the imp9 window to close
        imp9.hide()
        impB.hide()
    imp9.close()
    impR.close()
    impG.close()
    impB.close()

#------------------------------------------------
# making PFFcorrected B image
def make_exPFFc_B_image(impG, impB, parm_FFC_sigma, Mode, L_max, \
    folder, filename, parm_subtractBv, parm_leveladj, parm_correct_fac, \
                                                parm_ReadMask, debugMode):
    #   always execute pseudo_Flat_field_C for making mask
    imp_c_b_tmp, imp_Gauss_b1, imp_Gauss_g1 = \
        pseudo_Flat_field_C(impG, impB, parm_FFC_sigma, Mode, L_max, folder, \
            filename, parm_subtractBv, parm_leveladj, parm_correct_fac)
    imp_Gauss_b1.close()
    imp_Gauss_g1.close()

    #--   make blue higer image and combine imp_c_b
    if parm_ReadMask == True:
        impT = Duplicator().run(impB) # corrected B ch.
        impRf2 = Duplicator().run(impG) # G ch.

        impT_highlight, impRf_highlight = \
                make_imp_blueHigh(impRf2, impT, Mode, L_max)
        # make bluer region correction
        imp_c_b_high, imp_Gauss_b1_high, imp_Gauss_g1_high = \
            pseudo_Flat_field_C(impRf_highlight, impT_highlight, \
            parm_FFC_sigma, Mode, L_max, folder, filename, parm_subtractBv,\
            "No", 0.0)
            # option with parm_leveladj: "BlueAdjustment"
        ic = plugin.ImageCalculator()
        imp_c_b =  ic.run("Max create", imp_c_b_tmp, imp_c_b_high)
        imp_c_b.setDisplayRange(0.0, L_max)

        imp_c_b_tmp.close()
        imp_c_b_high.close()
        impT.close()
        impRf2.close()
    else:
        imp_c_b = imp_c_b_tmp
    if debugMode == True:
        statimp_c_b = imp_c_b.getStatistics()
        printStat(statimp_c_b,"RGB Manipuration stat imp_c_b: ")
    return imp_c_b

# -------- making impB03 image -----------
##      Make impB03 as B x 0.3 Image > B:G ratio variable
def make_impB03(bool_PFFcOnly, HybridC, imp_c_b, impB, Gmixrate, \
                debugMode, L_max):
    # imp_c_b: PFFc applyed B image
    # impB: B image
    # bool_PFFcOnly: PFFc applyed or not (True/False)
    # Gmixrate: G ch. mixing rate to B ch. (normally 0.7)
    # HybridC: Do hybric correction? (On/OFF)
    if bool_PFFcOnly == True: # EPFFc only
        impB03 = Duplicator().run(imp_c_b)
            # use extended pseudo FFc image
        Bmixrate = 1.0 - Gmixrate # B:G mixing rate variable
        strValueB =  "value=" + str(Bmixrate)
        strValueG =  "value=" + str(Gmixrate)
    elif (HybridC == "On") or (HybridC == "New"):
                    # Hybrid & New Hybrid algorithm
        impB03 = Duplicator().run(imp_c_b)
            # use extended pseudo FFc image
        strValueB =  "value=0.3" # B:G mixing rate fixed
        strValueG =  "value=0.7"
        Gmixrate = 0.7
    else: # existing algorithm
        impB03 = Duplicator().run(impB) # not to use FFcorection image
        strValueB =  "value=0.3"
        strValueG =  "value=0.7"
        Gmixrate = 0.7
            # use normal B image
    IJ.run(impB03, "Multiply...", strValueB)
    if Mode == "32-bit":
        if bool_PFFcOnly == True:
            impB03.setDisplayRange(0.0, L_max * Bmixrate)
        else:
            impB03.setDisplayRange(0.0, L_max * 0.3)
    if debugMode == True:
        statImpB03x03 = impB03.getStatistics()
        printStat(statImpB03x03,"RGB Manipuration stat multiplied impB03: ")
    return impB03, strValueB, strValueG, Gmixrate
#-----------------------------------------------
##        Make impG07 as G x 0.7 Image
def make_impG07(HybridC, imp_c_b, impG, strValueG, parm_FFC_sigma, Mode, \
                                    L_max, folder, filename, Gmixrate):
    # imp_c_b: PFFc applyed B image
    # impG: G image
    # parm_FFC_sigma: Sigma for Gaussian Blur for PFFc applyed to G ch.
    # Gmixrate / strValueG: G ch. mixing rate to B ch. (normally 0.7)
    # HybridC: Do hybric correction? (On/Off/New)
    # statB03: statistics of impB03
    if HybridC != "New":
        impG07 = Duplicator().run(impG)
        impG07.setDisplayRange(0.0, L_max)

    else: # HybridC == "New"
        imp_c_g, imp_Gauss_g2, imp_Gauss_b2 = \
            pseudo_Flat_field_C(imp_c_b, impG, parm_FFC_sigma * 0.5, Mode, \
            L_max, folder, filename, "True", "Auto", 1.0)
            # modified in ver. 5.01
            # parm_subtractBv: True / parm_leveladj: Auto /
            # parm_correct_fac 1.0 / parm_FFC_sigma: 5.0 (in HybridC == "New")

        impG07 = imp_c_g
        imp_Gauss_g2.close()
        imp_Gauss_b2.close()

    IJ.run(impG07, "Multiply...", strValueG)
    impG07.setDisplayRange(0.0, L_max * Gmixrate)

    return impG07
#-----------------------------------------------
#--- making foreground layer image
def making_Fg_image(impG, impB, parm_FFC_sigma, HybridC, Mode, L_max, \
    folder, filename, parm_subtractBv, parm_leveladj, parm_correct_fac, \
    bool_PFFcOnly, parm_Gmixrate, parm_ReadMask, Show, debugMode):
    # making extended pseudo Flat-Field corrected B image
    imp_c_b = make_exPFFc_B_image(impG, impB, parm_FFC_sigma, Mode, L_max, \
    folder, filename, parm_subtractBv, parm_leveladj, parm_correct_fac, \
                                                parm_ReadMask, debugMode)
    # imp_c_b: extended pseudo flat-field corrected B image
    #--
    ##      Make impB03 as B x 0.3 Image > B:G ratio variable
    impB03, strValueB, strValueG, Gmixrate = \
        make_impB03(bool_PFFcOnly, HybridC, imp_c_b, impB, \
                                parm_Gmixrate, debugMode, L_max)
    if Show == "Yes":
        impB03.show()
    # G ch. x 0.7
    ##        Make impG07 as G x 0.7 Image
    #    -----Next version modification will be here!
    impG07 = make_impG07(HybridC, imp_c_b, impG, \
        strValueG, parm_FFC_sigma, Mode, L_max, folder, \
                                         filename, Gmixrate)
    # adjustment mead differance of imp_c_b and impG
    if Show == "Yes":
        impG07.show()
    # B x 0.3 + G x 0.7
    #---  making foreground layer image & save
    ic = plugin.ImageCalculator()
    imp3 = ic.run("Add create", impB03, impG07) # No window made
    imp3.setDisplayRange(0.0, L_max)

    if debugMode == True:
        statImp3 = imp3.getStatistics()
        printStat(statImp3,"RGB Manipuration stat imp3: ")
    if Show == "Yes":
        impG07.hide()
    impG07.close()
    return imp3, impB03, imp_c_b

#---
def calculate_cent_mean_diff(impA, impB, folder, filename):
    """
    Calculate mean differance of 2 imps from cropped images
    """
    width = impA.getWidth()
    height = impA.getHeight()
    # size of impA and impB is same
    roi_x = int(width * 0.15)
    roi_y = int(height * 0.15)
    roi_width = int(width * 0.7)
    roi_height = int(height * 0.7)

    impA.setRoi(roi_x, roi_y, roi_width, roi_height)
    impB.setRoi(roi_x, roi_y, roi_width, roi_height)
    impA1 = impA.crop()
    impB1 = impB.crop()
    statA = impA1.getStatistics()
    statB = impB1.getStatistics()
    mean_diff = statA.mean - statB.mean
#### for test and debug
    #IJ.saveAsTiff(impA1, folder + filename + "_impA1cropped.tif")
    #IJ.saveAsTiff(impB1, folder + filename + "_impB1cropped.tif")
####
    impA1.close()
    impB1.close()
    impA.resetRoi()
    impB.resetRoi()
    return mean_diff

#---making bg mask--------------
def make_bg_mask(impOrg, lowerV, Mode, L_max):
    # add in Ver. 5.3a
    # make layer mask of background correction layer
    # lowerV: lowerThreshold / upperV: upperThreshold (double)
    # roiColor: Color Value, ROI filled with (double)
    # maskname: name of mask image (string)
    # impOrg: impR
    imp = Duplicator().run(impOrg)
    ip = imp.getProcessor()
    ip_threshold = ip.createProcessor(ip.width, ip.height)
    ip_threshold.setValue(((lowerV - 25.5) / 255.0) * L_max)
    ip_threshold.fill()
    impThreshold = ImagePlus("Bg Mask Threshold", ip_threshold)
    ic = plugin.ImageCalculator()
    impMask = ic.run("Subtract create", impOrg, impThreshold)
    multFac = 255.99 / 51.0
    impMask.getProcessor().multiply(multFac)
    if Mode == "32-bit":
        impMask = clipOverLmax(impMask, L_max)
    imp.close()
    impThreshold.close()
    return impMask

#--- making background layer image
# (G ch.+30) x 0.7
def making_Bg_image(impG, impB03, imp_c_b, bool_PFFcOnly, parm_Fg_tL, \
                                        bgG_gamma, parm_Gain, Show, L_max):
    # parm_Prph: threshold for periphery correction
    # bool_PFFcOnly: Do exPFFc only algorism? (Ture/False)
    # bgG_gamma: Gamma of Background layer to make image brighter
    # parm_Gain: adding value to make background layer brighter
    # parm_Fg_tL: Fg / Bg threshold = mid gray point

    ##    Make impG30x07 as (G+30) x 0.7 Image
    if bool_PFFcOnly == False:  # Applying Hybrid or Traditional algorism
        # Mixing lighter G image with B for background layer
        impG30x07 = Duplicator().run(impG)
        # Ver. 5.3 / following code applying for all bit depth image
        impG30x07.setDisplayRange(0.0, L_max)
        if Show == "Yes":
            impG30x07.show()
        # Gain for background layer image
        # appropriate gamma is /1.45(= 0.69) for perceptive TRC image
        if parm_Fg_tL == 128: # in case of perceptive gamma TRC image.
            IJ.run(impG30x07, "Gamma...", "value=" + str(bgG_gamma))
                                                    # originally 0.69
        else: # in case of linear image.
            IJ.run(impG30x07, "Gamma...", "value=" + str(bgG_gamma))
                                                    # originally 0.7
        impG30x07.setDisplayRange(0.0, L_max)
        IJ.run(impG30x07, "Multiply...", "value=0.7")
    ## B x 0.3 + (G+30) x 0.7 + x(parm_Gain)
        ic = plugin.ImageCalculator()
        imp4 = ic.run("Add create", impB03, impG30x07) # No window made
        imp4.setDisplayRange(0.0, L_max)

    #  add B x 0.3 + (G+30) x 0.7 + x
        IJ.run(imp4, "Add...", "value=" + str((parm_Gain / 255.0) * L_max))
                                        # add Value parm_Gain
        if Show == "Yes":
     #Hide the image windows to close
            impG30x07.hide()
        impG30x07.close()
    else:  # Apply Flat-field correction (exclude hybird)
        # No G ch. mixing in background layer
        imp4 = Duplicator().run(imp_c_b)
    return imp4
#-------------------------------------------------
#--- making Periphery Adjustment Mask
# Periphery Adjustment Mask: B - ((R+G)/2-10)
def making_Prph_adj_Mask(imp5, impB, parm_Prph, bool_PFFcOnly, \
                                                    Mode, L_max, Show):
    # imp5: (impG + impR) / 2
    # parm_Prph: threshold for periphery correction
    # bool_PFFcOnly: Do exPFFc only algorism? (Ture/False)
    # Mode: bitdepth
    # L_max: maximum value of image
    impGR = Duplicator().run(imp5) #imp5: (G+R)/2 image
    impGR.setDisplayRange(0.0, L_max)

    if Show == "Yes":
        impGR.show()
    IJ.run(impGR, "Subtract...", "value="+ str((parm_Prph / 255.0) * L_max))
                                            # subtract Value parm_Prph /10
    if Mode == "32-bit":
        if Show == "Yes":
            impGR.hide()
        impGR = clipUnder0(impGR, Mode, L_max)
    ic = plugin.ImageCalculator()
    if bool_PFFcOnly == False: # traditional & Hybrid algorism
        imp2GR = ic.run("Subtract create", impB, impGR)
                                    #impB - impGR:(G+R/2)
    else: # EPFFc only algorism
        imp2GR = ic.run("Subtract create", impB, impGR)
                            #corrected impB - impGR:(G+R/2)
    if Mode == "32-bit":
        imp2GR = clipUnder0(imp2GR, Mode, L_max)
    if Show == "Yes":
        imp2GR.show()
        impGR.hide()
    IJ.run(imp2GR, "Multiply...", "value=4") # Save the Subtracted image x 4
    impGR.close()
    return imp2GR

#------ making mask of Fg brown correction layer ------
def making_diff_RG_mask(imp3, impR, impG, lowerV, L_max, Mode, folder, filename):
    # making mask of R - G for brown corrction layer
    # imp3 corrected B ch. image
    # imp5 (impR + impG) /2
    # imp9 mask of Bg correction layer
    lowerV = lowerV / 255.0 # lowerV is originally 8bit based data
    impTmpR = Duplicator().run(impR)
    impTmpV = Duplicator().run(impR)
    impTmpG = Duplicator().run(impG)
    impTmpB = Duplicator().run(imp3)

    # --- make V(Lightness) mask ---
    ip = impTmpV.getProcessor()
    ip_threshold = ip.createProcessor(ip.width, ip.height)
    ip_threshold.setValue(lowerV * L_max)
    ip_threshold.fill()
    impThreshold = ImagePlus("Bg Mask Threshold", ip_threshold)
    ic = plugin.ImageCalculator()
    impMaskV = ic.run("Subtract create", impTmpV, impThreshold)
    multFac = 1.0 / 0.2
    impMaskV.getProcessor().multiply(multFac)
    if Mode == "32-bit":
        impMaskV = clipOverLmax(impMaskV, L_max)
    impTmpV.close()
    impThreshold.close()
    #impMaskV.getProcessor().invert()
    impMaskV = invert(impMaskV, Mode, L_max)


    # --- make R - G mask ---
    meanDiff = calculate_cent_mean_diff(impTmpR, impTmpG, folder, filename)
    # meanDiff: impR.mean - impG.mean
    impTmpR.getProcessor().subtract(meanDiff)
    mask0 = ic.run("Subtract create", impTmpR, impTmpG) # region B < R+G
    if Mode == "32-bit":
        mask0 = clipUnder0(mask0, Mode, L_max)
    mask = ic.run("Min create", impMaskV, mask0) # region B < R+G
    impTmpR.close()
    impTmpG.close()
    impTmpB.close()
    impMaskV.close()
    mask0.close()
    stat = mask.getStatistics()
    factor = (L_max * 0.4) / stat.mean
    mask.getProcessor().multiply(factor)
        # adjusting mask lightness mean value to L_max * 0.4
    return mask

#---------------------------------------------------
def making_brownC_layer(imp, Mode, L_max):
    # imp: fg correcting layer
    impTmp0 = Duplicator().run(imp)
    impTmp1 = sRGB_TRC_decode(impTmp0, L_max)
    if Mode == "32-bit":
        impTmp1 = clipUnder0(impTmp1, Mode, L_max)
    impTmp0.close()
    impTmp1.getProcessor().subtract(0.05 * L_max)
    impTmp2 = sRGB_TRC_encode(impTmp1, L_max)
    if Mode == "32-bit":
        impTmp2 = clipUnder0(impTmp2, Mode, L_max)
    impTmp1.close()
    return impTmp2

#----

def sRGB_TRC_encode(imp, L_max):
    # revised 2024.6
    # if imp_Shadow max value > shadow threshold: only use imp_nonShadow 
    stat0 = imp.getStatistics()
    impTrc1 = Duplicator().run(imp)
    impTrc2 = Duplicator().run(imp)
    cshadowThld = 0.0031308 * L_max
    # non shadow region
    impmask = imageBinarySmpl(impTrc1, cshadowThld, L_max, L_max, \
                          "non_shadow_mask")
    impTrc1.setDisplayRange(0.0, L_max)
    impTrc1.getProcessor().gamma(1.0 / 2.4)
    impTrc1.getProcessor().multiply(1.055)
    impTrc1.getProcessor().subtract(0.055 * L_max)
    ic = plugin.ImageCalculator()
    imp_nonShadow =  ic.run("Min create", impmask, impTrc1)

    # shadow region
    if stat0.min <= cshadowThld:
        impmaskShadow = imageBinarySmpl(impTrc2, 0.0, cshadowThld, L_max, \
                                    "shadow_mask")
        impTrc2.setDisplayRange(0.0, L_max)
        impTrc2.getProcessor().multiply(12.92)
        imp_Shadow =  ic.run("Min create", impmaskShadow, impTrc2)
        impmaskShadow.close()

    else: # if imp minvalue > cshadowThld
        imp_Shadow = impTrc2.duplicate()
        imp_Shadow.getProcessor().setValue(0.0)
        imp_Shadow.getProcessor().fill()

    # shadow + non shadow
    impTrc3 =  ic.run("Max create", imp_Shadow, imp_nonShadow)
    impTrc3.setTitle("sRGB TRC encoded")

    impmask.close()
    imp_Shadow.close()
    imp_nonShadow.close()
    impTrc1.close()
    impTrc2.close()
    return impTrc3

#-----------
def sRGB_TRC_decode(imp, L_max):
    # revised 2024.6
    # if imp_Shadow max value > shadow threshold: only use imp_nonShadow 
    stat0 = imp.getStatistics()
    impTrc1 = Duplicator().run(imp)
    impTrc2 = Duplicator().run(imp)
    cshadowThld = 0.04045 * L_max

    # not shadow region
    impmask = imageBinarySmpl(impTrc1, cshadowThld, L_max, L_max, \
                          "shadow_mask")
        # ROI: white back: black
    impTrc1.setDisplayRange(0.0, L_max)
    impTrc1.getProcessor().add(0.055 * L_max)
    impTrc1.getProcessor().multiply(1 / 1.055)
    impTrc1.getProcessor().gamma(2.4)
    ic = plugin.ImageCalculator()
    imp_nonShadow =  ic.run("Min create", impmask, impTrc1)
    # shadow region
    if stat0.min <= cshadowThld:
        impmaskShadow = imageBinarySmpl(impTrc2, 0.0, cshadowThld, L_max, \
                                "non_shadow_mask")
        impTrc2.setDisplayRange(0.0, L_max)
        impTrc2.getProcessor().multiply(1 / 12.92)
        imp_Shadow =  ic.run("Min create", impmaskShadow, impTrc2)
        impmaskShadow.close()
    else: # if imp minvalue > cshadowThld
        imp_Shadow = impTrc2.duplicate()
        imp_Shadow.getProcessor().setValue(0.0)
        imp_Shadow.getProcessor().fill()

    stat = imp_Shadow.getStatistics()
    if stat.max < cshadowThld:
        impTrc3 =  ic.run("Max create", imp_Shadow, imp_nonShadow)
    else:
        impTrc3 =  imp_nonShadow.duplicate()
    impmask.close()
    imp_Shadow.close()
    imp_nonShadow.close()
    impTrc1.close()
    impTrc2.close()

    return impTrc3



#-------------------------------------------------
class DialogPreviewer(ItemListener, AdjustmentListener):
    def __init__(self, choice, slider, nfield, \
                            nfield1, nfield3, nfield2=None):
        """
          ##choice-TRC > parameter fields
          ##choice-Hybrid correction (novice)> parameter fields
            choice: a java.awt.Choice
            processtype: string
            nfield:  a java.awt.TextField
            slider:  a java.awt.Scrollbar

        """
        self.slider = slider # Mixing G ch. Gamma
        self.nfield = nfield # Background mask threshold
        self.nfield1 = nfield1 # Shadow area threshold
        self.nfield2 = nfield2 # Periphery threshold
        self.nfield3 = nfield3 # slider numeric field
        # default: self.nfield3.setText("1.56")
        # default: self.slider.setValue(156) # 1.56
        self.choice1 = choice
        #IJ.showMessage("Value set: " + str(self.slider.getValue()))


    def itemStateChanged(self, event):
        """ event: an ItemEvent with data on what happened to the checkbox. """
        choice_item  = self.choice1.getSelectedItem()
        print "itemStateChanged", str(choice_item)
        if choice_item == "Perceptual": # Select TRC
            self.nfield.setText("128")
            self.nfield1.setText("25")
            if self.nfield2 != None:
                self.nfield2.setText("10")
                self.nfield2.setBackground(Color.white)
            self.nfield.setBackground(Color.white)
            self.nfield1.setBackground(Color.white)
        elif choice_item == "Linear": # Select TRC
            self.nfield.setText("128")
            self.nfield1.setText("25")
            if self.nfield2 != None:
                self.nfield2.setText("0")
                self.nfield2.setBackground(Color.cyan)
            self.nfield.setBackground(Color.cyan)
            self.nfield1.setBackground(Color.cyan)
        elif choice_item == "Off": # Select Hypbid Correction
            self.nfield3.setText("1.45")
            self.slider.setValue(145) #1.46
            self.nfield3.setBackground(Color.pink)
        elif choice_item == "On" or choice_item == "New Method On" :
                                                # Select Hypbid Correction
            self.nfield3.setText("1.56")
            #IJ.showMessage("Value set: " + str(self.slider.getValue()))
            self.slider.setValue(156) # 1.56
            self.nfield3.setBackground(Color.white)

    def adjustmentValueChanged(self, event):
        """ event: an AdjustmentEvent with data on the state of the scroll bar. """
        value = self.nfield3.getText()
        if value == "1.00":
            self.nfield3.setBackground(Color.lightGray)
        elif value == "1.45":
            self.nfield3.setBackground(Color.pink)
        elif value == "1.56":
            self.nfield3.setBackground(Color.white)
        else:
            self.nfield3.setBackground(Color.yellow)
        return
#------------------------------------------------
class DialogPreviewer1 (ItemListener):
    def __init__(self, choice, choice2, slider, nfield,  nfield1, \
                                        nfield3, nfield4, nfield5, nfield6):
        """
           ##choice-Hybrid correction (expert)> parameter fields
             choice: a java.awt.Choice
             nfield:  a java.awt.TextField
             slider:  a java.awt.Scrollbar
        """
        self.slider = slider # Mixing G ch. Gamma
        self.nfield = nfield # Background mask threshold
        self.nfield1 = nfield1 # Shadow area threshold
        self.nfield3 = nfield3 # G's Gamma slider numeric field
        self.choice1 = choice  # Hybrid correction
        self.nfield4 = nfield4 # Sigma
        self.nfield5 = nfield5 # G mix ratio
        self.nfield6 = nfield6 # incremental amount
        self.choice2 = choice2 # incremental amount on/off
        self.choice_item2  = self.choice2.getSelectedItem()

    def itemStateChanged(self, event):
        """ event: an ItemEvent with data on what happened to the checkbox. """
        choice_item  = self.choice1.getSelectedItem()
        print "itemStateChanged", str(choice_item)
        if choice_item == "G Mixing":
                                        # Select Traditional algorism
            self.nfield3.setText("1.45")
            self.slider.setValue(145) #1.45
            self.nfield3.setBackground(Color.pink)
            self.nfield4.setBackground(Color.gray) # Sigma
            self.nfield5.setBackground(Color.gray) # G mix ratio
            self.nfield6.setBackground(Color.gray) # incremental amount
        elif choice_item == "Hybrid" or choice_item == "New Hybrid" :
                                            # Select Hypbid Correction
            self.nfield3.setText("1.56")
            self.slider.setValue(156) # 1.56
            self.nfield3.setBackground(Color.white)
            self.nfield4.setBackground(Color.green) # Sigma
            if choice_item == "New Hybrid":
                self.nfield4.setText("5.0")
            self.nfield5.setBackground(Color.gray) # G mix ratio
            if self.choice_item2  == "Manual":
                self.nfield6.setBackground(Color.green)
        elif choice_item == "EPFFc":
                        # Select EPFFc
            self.nfield3.setText("1.00")
            self.slider.setValue(100) # 1.00
            self.nfield3.setBackground(Color.lightGray)
            self.nfield4.setBackground(Color.green) # Sigma
            self.nfield5.setBackground(Color.green) # G mix ratio
            if self.choice_item2  == "Manual":
                self.nfield6.setBackground(Color.green)
        return
#------------------------------------------------
class DialogPreviewer2(ItemListener):
    def __init__(self, choice, nfield3, nfield4, nfield5, \
                                    nfield6, choice2, choice3, checkbox2):
        """
           ##checkbox-pFF correction > parameter fields
             choice: a java.awt.Choice
             checkbox:  a java.awt.Checkbox
        """
        self.choice1 = choice # Hybrid c on/off
        choice_item1  = self.choice1.getSelectedItem()
        self.nfield3 = nfield3 # Mixing G ch. Gamma: slider's numeric field
        self.nfield4 = nfield4 # Sigma
        self.nfield5 = nfield5 # G mix ratio
        self.nfield6 = nfield6 # incremental amount
        self.choice2 = choice2 # incremental amount on/off
        choice_item2  = self.choice2.getSelectedItem()
                            # adjustment amount auto / manual / none
        self.choice3 = choice3 # imege conditon
        choice_item3  = self.choice3.getSelectedItem()
        self.checkbox2 = checkbox2 # Apply pre-mixing to B
        if choice_item1 == "EPFFc": # apply FF correction only
            self.nfield4.setBackground(Color.green)
            self.nfield5.setBackground(Color.green)
            if choice_item2  == "Manual":
                self.nfield6.setBackground(Color.green)
            else:
                self.nfield6.setBackground(Color.gray)
        elif  choice_item1 == "Hybrid" or  choice_item1 == "New Hybrid":
                                                            # apply Hybrid
            self.nfield4.setBackground(Color.green)
            self.nfield5.setBackground(Color.gray)
            if choice_item2  == "Manual":
                self.nfield6.setBackground(Color.green)
            else:
                self.nfield6.setBackground(Color.gray)
        else:  # G mixing only
            self.nfield4.setBackground(Color.gray)
            self.nfield5.setBackground(Color.gray)
            self.nfield6.setBackground(Color.gray)
        if nfield3.getText() == "1.00": # if Gamma slider value is 100
            self.nfield3.setBackground(Color.lightGray)
        elif nfield3.getText() == "1.45": # if Gamma slider value is 145
            self.nfield3.setBackground(Color.pink)
        elif nfield3.getText() == "1.56": # if Gamma slider value is 145
            self.nfield3.setBackground(Color.white)
        else:
            self.nfield3.setBackground(Color.yellow)
        if choice_item3 == "Normal": # image conditionp
            self.nfield4.setText("10.0") # set Sigma value
            self.checkbox2.setState(False)
        else: # "Coarse"
            choice_item1  = self.choice1.getSelectedItem()
            if choice_item1 != "G Mixing":
                self.nfield4.setText("5.0")

    def itemStateChanged(self, event):
        """ event: an ItemEvent with data on what happened to the checkbox. """
        choice_item1  = self.choice1.getSelectedItem()
                                        # algorism selection
        choice_item2  = self.choice2.getSelectedItem()
                                    # adjustment amount auto / manual / none
        choice_item3  = self.choice3.getSelectedItem()
                                         # image condition normal / coarse

        if choice_item2 == "Manual": # pseudo FF correction On
            self.nfield6.setBackground(Color.green)
        else:
            self.nfield6.setBackground(Color.gray)
        if choice_item3 == "Normal": # image conditionp
            if choice_item1 != "New Hybrid":
                self.nfield4.setText("10.0") # set Sigma value
            self.checkbox2.setState(False)
        else: # "Coarse"
            choice_item1  = self.choice1.getSelectedItem()
            if choice_item1 != "G Mixing":
                self.nfield4.setText("5.0")
            self.checkbox2.setState(True)
        return

#------------------------------------------------
# Novice Mode: Get parameters from the input DialogBox
def NoviceMenu(path, Vmean):
    gd = GenericDialog("Novice Menu: Enter Parameters")
    TRCType = ["Perceptual", "Linear"]
    gd.addChoice("Image TRC Type", TRCType, TRCType[0])
    if Vmean < 128.0:
        BkThs = 128.0
    else:
        BkThs = int(Vmean)
    gd.addNumericField("Background Mask Threshold (0 to 255)", int(BkThs) , 0)
    # gd.addNumericField("Background Image Gain (Max to 100)", 0, 0)
    gd.addSlider("Background mixing G ch. Gamma (increase > bluer)",\
                                                    1.00, 1.70, 1.56 ,0.01)
    gd.addNumericField("Shadow Area Threshold (0 to 255)", 25, 0)
    HybridC_Choice = ["Off", "On", "New Method On"]
    gd.addChoice("Apply hybrid correction?", \
                                        HybridC_Choice, HybridC_Choice[1])
    #gd.addCheckbox("(in Hybrid correction) Preserve plants green", False)
    condition_Choice = ["Normal","Coarse"]
    gd.addChoice("Image condition", condition_Choice, condition_Choice[0])
    showWin = ["Yes","No"]
    gd.addChoice("Show working windows?", showWin, showWin[0])
    gd.addMessage("If you would like to use Expert Menu,"+\
                                            " Please cancel this menu.")
    gd.addMessage("Recommended Parameters for Linear Images\n"+ \
        " 56 for Background Mask Threshold\n 3 for Shadow Area Threshold\n" +\
        " 0 for Periphery Adjst Threshold")

    # The UI elements for the above inputs
    choice = gd.getChoices().get(0) # TRC Type
    nfield = gd.getNumericFields().get(0) # Background mask threshold
    slider = gd.getSliders().get(0) # Mixing G Gamma
    nfield3 = gd.getNumericFields().get(1) # slider numeric field
    # Attention! Sliders has also NumericFields, so attention
    #                                   to index number for NFields!
    nfield1 = gd.getNumericFields().get(2) # Shadow area threshold
    choice1 = gd.getChoices().get(1) # Hybrid correction on/off
    # Set event previewer
    previewer = DialogPreviewer(choice, slider, nfield, \
                                        nfield1, nfield3)
    previewer1 = DialogPreviewer(choice1, slider, nfield, \
                                        nfield1, nfield3)
    choice.addItemListener(previewer)
    slider.addAdjustmentListener(previewer)  # Slider
    choice1.addItemListener(previewer1)

    gd.showDialog()

    # get values
    parm_Mode   = "Auto"
    TRCselected   = gd.getNextChoice()
    if TRCselected == "Perceptual":
        parm_Fg_tL = 128
        parm_Prph = 10
    else:
        parm_Fg_tL = 56
        parm_Prph = 0
    parm_Binary   = gd.getNextNumber() # Fg / Bg threshold
    parm_Gain = gd.getNextNumber() # (5.01a) Gamma value for Bg
    parm_Shadow = gd.getNextNumber() # Shadow correction threshold
    parm_HybridC   = gd.getNextChoice() # Correction method
    if parm_HybridC == "New Method On":
        parm_HybridC = "New"
    #green_preserve = gd.getNextBoolean() # apply green preserve True / False
    parm_Condition   = gd.getNextChoice() # Image condition Normal / Coarse
    parm_Show   = gd.getNextChoice() # Show windows?
    # following parameters are fixed in Novice Mode
    bool_PFFcOnly = False # apply only pseudo flat-field correction > false
    if parm_Condition == "Normal" and parm_HybridC !="New":
        parm_FFC_sigma = 10.0
        DoPremixing = False
    else: # image condition: "Coarse" or parm_HybridC: "New"
        parm_FFC_sigma = 5.0
        DoPremixing = True
    parm_Gmixrate = 0.0
    parm_leveladj  = "Auto" # apply level adjustment between channels
    parm_correct_fac = 0.0 # correction adjustment factor
    parm_subtractBv = False
    bgG_gamma = 1.0 / parm_Gain # inverse value of gamma (actual applying power)
    parm_Gain = 0.0
    if parm_Binary > 255: #Set Max Value
        parm_Binary = 255
    if parm_Gain > 1.7:
        parm_Gain = 1.7
    if parm_Shadow > 255: #Set Max Value
        parm_Shadow = 255
    """if green_preserve == True: #changed in 4.82
        Fg_mask_type = "Type4"
    elif green_preserve == False:"""
    Fg_mask_type = "Type2"
    parm_ReadMask = False
    if gd.wasCanceled():
        choice.removeItemListener(previewer)
        choice1.removeItemListener(previewer1)
        parm_Mode   = "Expert"
        IJ.log("Expert Menu called")
        return str(parm_Mode), parm_Binary, parm_Gain, parm_Shadow,\
            str(parm_Show), parm_Prph, parm_Fg_tL, bool_PFFcOnly, \
            parm_FFC_sigma, parm_Gmixrate, parm_subtractBv, parm_leveladj, \
            parm_correct_fac, parm_HybridC, Fg_mask_type, bgG_gamma, \
            parm_ReadMask, DoPremixing
    write_out(path, parm_Binary, parm_Gain, parm_Shadow, parm_Prph, \
            parm_Fg_tL, bool_PFFcOnly, parm_FFC_sigma, parm_Gmixrate, \
            parm_subtractBv, parm_leveladj, parm_correct_fac, parm_HybridC, \
            Fg_mask_type, bgG_gamma, parm_Condition)
    choice.removeItemListener(previewer)
    choice1.removeItemListener(previewer1)
    return str(parm_Mode), parm_Binary, parm_Gain, parm_Shadow, \
        str(parm_Show), parm_Prph, parm_Fg_tL, bool_PFFcOnly, \
        parm_FFC_sigma, parm_Gmixrate, parm_subtractBv, parm_leveladj, \
        parm_correct_fac, parm_HybridC, Fg_mask_type, bgG_gamma, \
        parm_ReadMask, DoPremixing
#----------------------------------------------------
# Expert Mode: Get parameters from the input DialogBox
def getParameters(path, Vmean):
    oldparms = read_in(path, Vmean)
    gd = GenericDialog("Expert Menu: Enter Parameters")
    TRCType = ["Perceptual", "Linear"]
    if int(oldparms["parm_Fg_tL"]) == 128:
        gd.addChoice("Image TRC Type", TRCType, TRCType[0])
    else:
        gd.addChoice("Image TRC Type", TRCType, TRCType[1])

    gd.addNumericField("Background Mask Threshold (0 to 255)",\
                                            float(oldparms["parm_Binary"]), 0)
    gd.addSlider("Background mixing G ch. Gamma (increase > bluer)", \
                        1.00, 1.70, 1.00 / float(oldparms["bgG_gamma"]), 0.01)
    gd.addNumericField("Shadow Area Threshold (0 to 255)", \
                                            float(oldparms["parm_Shadow"]), 0)
    gd.addNumericField("Periphery Adjst Threshold (-10 to 20)", \
                                            float(oldparms["parm_Prph"]), 0)

    gd.addMessage("                        in algorism /" + \
                    " EPFFc: Apply Extended Pseudo Flat-Field Correction only")

    HybridC_Choice = ["G Mixing", "Hybrid", "New Hybrid", "EPFFc"]
    if float(oldparms["bgG_gamma"]) == 0.69:
        gd.addChoice("Select algorism", HybridC_Choice, HybridC_Choice[0])
    elif float(oldparms["bgG_gamma"]) == 1.00:
        gd.addChoice("Select algorism", HybridC_Choice, HybridC_Choice[3])
    else:
        gd.addChoice("Select algorism", HybridC_Choice, HybridC_Choice[1])
    mask_type = ["Type1", "Type2", "Type3", "Type4","None"]
    """if oldparms["Fg_mask_type"] == "Type1":
        gd.addChoice("Mask type for foreground correction layer", \
                                            mask_type, mask_type[0])
    elif oldparms["Fg_mask_type"] == "Type3":
        gd.addChoice("Mask type for foreground correction layer", \
                                            mask_type, mask_type[2])
    elif oldparms["Fg_mask_type"] == "Type4":
        gd.addChoice("Mask type for foreground correction layer", \
                                            mask_type, mask_type[3])
    elif oldparms["Fg_mask_type"] == "None":
        gd.addChoice("Mask type for foreground correction layer", \
                                            mask_type, mask_type[4])
    else:
        gd.addChoice("Mask type for foreground correction layer", \
                                            mask_type, mask_type[1])
    gd.addMessage("       Type1: R+128 mask " + \
          "       Type2: Preserve Lighter B region " \
        + "\n       Type3: Preserve plants green        Type4: Type2 + 3")"""
    gd.addMessage("-----------------------------------")
    gd.addMessage("[Extended Pseudo Flat-Field correction (EPFFc) parameters]")
    gd.addNumericField("Sigma for Flat-Field correction", \
                                    float(oldparms["parm_FFC_sigma"]), 1)
    gd.addNumericField("G mix rate to B (available in EPFFc only mode)", \
                                    float(oldparms["parm_Gmixrate"]), 1)
    leveladj = ["No","Auto","Manual"]
    if oldparms["parm_leveladj"] == "Auto":
        gd.addChoice("Correction Amount adjustment", leveladj, leveladj[1])
    elif oldparms["parm_leveladj"] == "Manual":
        gd.addChoice("Correction Amount adjustment", leveladj, leveladj[2])
    else:
        gd.addChoice("Correction Amount adjustment", leveladj, leveladj[0])
    gd.addNumericField("Incremental correcting amount (in Manual 0-255)", \
                                    float(oldparms["parm_correct_fac"]), 0)
    gd.addCheckbox("Subtract Pixel Value if B > G",\
                                        eval(oldparms["parm_subtractBv"]))
    gd.addCheckbox("Read user made bluer mask", False)
    gd.addCheckbox("Apply G texture info to original B ", True)
    gd.addMessage("-----------------------------------")
    condition_Choice = ["Normal","Coarse"]
    if oldparms["parm_Condition"] == "Normal":
        gd.addChoice("Image condition", condition_Choice, condition_Choice[0])
    elif oldparms["parm_Condition"] == "Coarse":
        gd.addChoice("Image condition", condition_Choice, condition_Choice[1])
    showWin = ["Yes","No"]
    gd.addChoice("Show working windows?", showWin, showWin[0])

    # The UI elements for the above inputs
    choice = gd.getChoices().get(0) # TRC Type
    nfield = gd.getNumericFields().get(0) # Background mask threshold
    slider = gd.getSliders().get(0) # Mixing G Gamma
    nfield3 = gd.getNumericFields().get(1) # slider numeric field
    # Attention! Sliders has also NumericFields,
    #           so attention to index number for NFields!
    nfield1 = gd.getNumericFields().get(2) # Shadow area threshold
    nfield2 = gd.getNumericFields().get(3) # Periphery threshold
    choice1 = gd.getChoices().get(1) # Hybrid correction on/off
    #choice2: mask type choice > deleted!
    choice3 = gd.getChoices().get(2) # increment adjustment
    choice4 = gd.getChoices().get(3) # image condition
    nfield4 = gd.getNumericFields().get(4) # Sigma for FF correction
    nfield5 = gd.getNumericFields().get(5) # G mix ratio
    nfield6 = gd.getNumericFields().get(6) # Incremantarl correction amount
    #checkbox0: Subtract Pixcel Value if B > G
    #checkbox1: Using user made bluer mask
    checkbox2 = gd.getCheckboxes().get(2) # Apply pre mixing to B


    # Set event previewer
    #  TRC selection
    previewer = DialogPreviewer(choice, slider, nfield, \
                                        nfield1, nfield3, nfield2)
    #  Hybric correction
    previewer1 = DialogPreviewer1(choice1, choice3, slider, nfield, \
                                nfield1, nfield3, nfield4, nfield5, nfield6)
    #  Apply extended pseudo FF correction
    previewer2 = DialogPreviewer2(choice1, nfield3, nfield4, \
                        nfield5, nfield6, choice3, choice4, checkbox2)

    choice.addItemListener(previewer)  # TRC Type
    slider.addAdjustmentListener(previewer)  # Slider
    choice1.addItemListener(previewer1)  # Hybrid correction on/off
    # checkbox.addItemListener(previewer2)  # Apply FF correction
    choice3.addItemListener(previewer2)  # Apply FF correction - Manual
    choice4.addItemListener(previewer2)  # Apply FF correction - Manual

    gd.showDialog()

    #Get Parameters
    parm_Mode   = "Auto"
    TRCselected   = gd.getNextChoice()
    if TRCselected == "Perceptual":
        parm_Fg_tL = 128
    else:
        parm_Fg_tL = 56
    parm_Binary   = gd.getNextNumber() # Fg / Bg threshold
    parm_Gain = gd.getNextNumber() # (5.01a) Gamma value for Bg
    parm_Shadow = gd.getNextNumber() # Shadow correction threshold
    parm_Prph = gd.getNextNumber() # Peripheral fading correction threshold
    parm_HybridC   = gd.getNextChoice() # Correction method
    if parm_HybridC == "New Hybrid":
        parm_HybridC = "New"
        bool_PFFcOnly = False
    elif parm_HybridC == "G Mixing":
        parm_HybridC = "Off"
        bool_PFFcOnly = False
    elif parm_HybridC == "Hybrid":
        parm_HybridC = "On"
        bool_PFFcOnly = False
    elif parm_HybridC == "EPFFc":
        parm_HybridC = "Off"
        bool_PFFcOnly = True
    Fg_mask_type  = "Type2" # Fg layer mask type
    # bool_PFFcOnly = gd.getNextBoolean() #get value of checkbox
    parm_FFC_sigma = gd.getNextNumber()
                # Sigma for Gaussian filtering for pseudo FF correction
    parm_Gmixrate = gd.getNextNumber() # G mixing rate for B
    parm_leveladj  = gd.getNextChoice() # apply level adjustment between channels?
    parm_correct_fac = gd.getNextNumber() # correction adjustment factor
    parm_subtractBv = gd.getNextBoolean() #get value of checkbox
    parm_ReadMask = gd.getNextBoolean() # read user made mask?
    DoPremixing = gd.getNextBoolean() # apply premixing for original B image?
    parm_Condition   = gd.getNextChoice() # Image condition Normal / Coarse

    parm_Show   = gd.getNextChoice() # show correcting process?
    bgG_gamma = 0.69
    if parm_Binary > 255: #Set Max Value
        parm_Binary = 255
    if parm_Gain > 1.7:
        parm_Gain = 1.7
    bgG_gamma = 1.0 / parm_Gain
    parm_Gain = 0.0
    if parm_Shadow > 255: #Set Max Value
        parm_Shadow = 255
        if parm_Gmixrate > 1.0:
            parm_Gmixrate = 1.0
        elif parm_Gmixrate < 0.0:
            parm_Gmixrate = 0.0
    if gd.wasCanceled():
        parm_Mode   = "Stop"
        IJ.log("Canceled")
        return str(parm_Mode), parm_Binary, parm_Gain, parm_Shadow,\
            str(parm_Show), parm_Prph, parm_Fg_tL, bool_PFFcOnly, \
            parm_FFC_sigma, parm_Gmixrate, parm_subtractBv, parm_leveladj, \
            parm_correct_fac, parm_HybridC, Fg_mask_type, \
            bgG_gamma, parm_ReadMask, DoPremixing
    write_out(path, parm_Binary, parm_Gain, parm_Shadow, parm_Prph, \
            parm_Fg_tL, bool_PFFcOnly, parm_FFC_sigma, parm_Gmixrate, \
            parm_subtractBv, parm_leveladj, parm_correct_fac, parm_HybridC, \
            Fg_mask_type, bgG_gamma, parm_Condition)
    return str(parm_Mode), parm_Binary, parm_Gain, parm_Shadow, \
        str(parm_Show), parm_Prph, parm_Fg_tL, bool_PFFcOnly, \
        parm_FFC_sigma, parm_Gmixrate, parm_subtractBv, parm_leveladj, \
        parm_correct_fac, parm_HybridC, Fg_mask_type, bgG_gamma, \
        parm_ReadMask, DoPremixing
#----------------
def write_out(path, parm_Binary, parm_Gain, parm_Shadow, parm_Prph, \
            parm_Fg_tL, bool_PFFcOnly, parm_FFC_sigma, parm_Gmixrate, \
            parm_subtractBv, parm_leveladj, parm_correct_fac, parm_HybridC, \
            Fg_mask_type, bgG_gamma, parm_Condition):
    wo_parm = dict(parm_Binary=parm_Binary, parm_Gain=parm_Gain, \
        parm_Shadow=parm_Shadow, parm_Prph=parm_Prph, \
        parm_Fg_tL=parm_Fg_tL, bool_PFFcOnly=bool_PFFcOnly, \
        parm_FFC_sigma=parm_FFC_sigma, parm_Gmixrate=parm_Gmixrate, \
        parm_subtractBv=parm_subtractBv, parm_leveladj=parm_leveladj, \
        parm_correct_fac=parm_correct_fac, parm_HybridC=parm_HybridC, \
        Fg_mask_type=Fg_mask_type, bgG_gamma=bgG_gamma, \
        parm_Condition=parm_Condition)
#  -- writing "filename_parameter.txt" file
    writing_file0 = open(path  + '_parameter.txt','a')
    # new_file_number = int(file_number) + 1
    # i = 0
    for key in wo_parm.keys():
        writing_file0.write(key + ": " + str(wo_parm[key]) + "\n")
    timenow = datetime.datetime.now()
    writing_file0.write("date time: " + \
        timenow.strftime('%Y/%m/%d/%H:%M:%S') + "\n")
    writing_file0.close()
#-------------
def read_in(path, Vmean):
    existing_parms = {}
        # define dic type Variable
    # set default value
    existing_parms["parm_FFC_sigma"] = 10.0
    existing_parms["parm_subtractBv"] = "False"
    existing_parms["parm_leveladj"] = "Auto"
    existing_parms["parm_Shadow"] = 25.0
    existing_parms["parm_Prph"] = 10.0
    existing_parms["parm_correct_fac"] = 0.0
    if Vmean < 128.0:
        BkThs = 128.0
    else:
        BkThs = int(Vmean)

    existing_parms["parm_Binary"] = BkThs
    existing_parms["parm_Gmixrate"] = 0.0
    existing_parms["parm_Gain"] = 0.0
    existing_parms["bool_PFFcOnly"] = "False"
    existing_parms["parm_Fg_tL"] = 128
    existing_parms["parm_HybridC"] = "Off"
    existing_parms["Fg_mask_type"] = "Type2"
    existing_parms["bgG_gamma"] = 0.64
    existing_parms["parm_Condition"] = "Normal"
    existing_parms["existing"] = "No"
    is_file = os.path.isfile(path + '_parameter.txt')
    if is_file:
        # reading existing parameters
        with open(path + '_parameter.txt','r') as read_file:
            lines = read_file.readlines()
            if len(lines) >= 15:
                for l in range(len(lines)-15, len(lines)-1):
                    eachline = lines[l].strip()
                    i = eachline.find(":")
                    key = eachline[:i]
                    value = eachline[i+2:]
                    existing_parms[key] = value
                    existing_parms["existing"] = "Yes"
    # print "existing: ", existing_parms["existing"], "Lines: ", len(lines)

    return existing_parms

# ------ B level adjustment ----------
def B_level_adjustment(impsCh, Mode, L_max, TRCType):
    impsCht2 = []
    for i in range(3):
        if TRCType == "Perceptual":
            impsCht2.append(sRGB_TRC_decode(Duplicator().run(impsCh[i]), L_max))
        else:
            impsCht2.append(Duplicator().run(impsCh[i]))
    if Mode == "16-bit":
        for i in range(3):
            ImageConverter(impsCht2[i]).convertToGray16()
    elif Mode == "32-bit":
        for i in range(3):
            ImageConverter(impsCht2[i]).convertToGray32()
    statR = get_cropped_imp_stat(impsCht2[0])
    statG = get_cropped_imp_stat(impsCht2[1])
    statB = get_cropped_imp_stat(impsCht2[2])
    RGmean = (statR.mean + statG.mean) / 2.0
    Conversion = False
    if RGmean * 0.7  > statB.mean:
        Conversion = IJ.showMessageWithCancel("Very yellowed image was detected.",\
                                              "B level will be adjusted automatically.")
        if Conversion == True:
            factor = RGmean * 0.8 / statB.mean
            impsCht2[2].getProcessor().multiply(factor)
            impsCht2[2].getProcessor().gamma(0.8)
            statB2 =  get_cropped_imp_stat(impsCht2[2])
            if statG.max > statB2.max:
                factor = 1.05 * statG.max / statB2.max
                impsCht2[2].getProcessor().multiply(factor)
    impsCht1 = []
    for i in range(2):
        impsCht1.append(impsCh[i])
    if Conversion == True:
        if TRCType == "Perceptual":
            impsCht1.append(sRGB_TRC_encode(impsCht2[2], L_max))
        else:
            impsCht1.append(impsCht2[2])
    else:
        impsCht1.append(impsCh[2])
    for i in range(3):
        impsCht2[i].close()

    return impsCht1

#-------------
# Main
CountN = 0
FileCount = 0
JobDo = True
window = WM.getActiveWindow()
imgCurrent = WM.getCurrentImage()
if not imgCurrent:
    od = OpenDialog("Choose a file")
    folder = od.getDirectory()
    file = od.getFileName()
    if file:
        path = folder + file
        IJ.log(path)
        IJ.log(folder)
        IJ.log(file)
        imps = BF.openImagePlus(path) # Open file with Bio-format Plugin
        imp0 = imps[0]
    else:
        IJ.log("Job Canceled")
        JobDo = False
else: # target image is already open
    impgot = IJ.getImage()
    imps = [impgot]
    file = imps[0].getTitle()
    try:
        folder = imps[0].getOriginalFileInfo().directory
    except:
        folder = IJ.getDirectory("Output_directory")
    path = folder + file
if JobDo == True:
    bitdepth = imps[0].getBitDepth()
    if bitdepth != 0:
        L_max = set_Lmax(imps[0])
        imps[0].setDisplayRange(0.0, L_max)
        stat0 = imps[0].getStatistics()
        Vmean = stat0.mean
            # Vmean: for culculating threshold of Bg collection layer
        if bitdepth == 16:
            Vmean = int(Vmean / 256)
            Mode0 = "16-bit"
        elif bitdepth ==32:
            Vmean = int((Vmean / 255.0) * L_max)
            Mode0 = "32-bit"
        else: # 8 or 24bit (colorProcessor)
            Mode0 = "8-bit"
        Vmean = Vmean + 3
        if bitdepth == 32:
            IJ.showMessage("32bit Max L Value: " + str(L_max))
        ####
#        B_level_adjustment(imp, BitDepth)
        #### calling parameter input dialogs
        parm = NoviceMenu(path, Vmean)
        if parm[0] == "Expert":
            parm = getParameters(path, Vmean)
        print ("Parm[0]=", parm[0])
        #### parameter input end
        if parm[0] != "Stop":
            if parm[0] == "Auto": # parm[0] = Mode
                Mode = Mode0 # set bitDepth info
            imps[0].show()
            if parm[6] == 56: # parm_Fg_tL (56: linear / 128: Perceptual)
                TRCType = "Linear"
            else:
                TRCType = "Perceptual"
            impR, impG, impB = RGBmanipulation\
                    (folder, file, parm, imps[0], L_max, Mode, TRCType)
            base(impR, impG, impB, folder, file, parm, L_max, Mode, TRCType)
            imps[0].close()
    else:
        IJ.showMessage("This image is not compatible with this program.")
IJ.log("Finish")
